
/*--------------------------------------------------------------------*/
/*--- A header file for all parts of the Redux skin.  rx_include.h ---*/
/*--------------------------------------------------------------------*/

/*
   This file is part of Redux, a Valgrind skin for tracing program
   execution.

   Copyright (C) 2003 Nicholas Nethercote
      njn25@cam.ac.uk

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/

#ifndef __RX_INCLUDE_H
#define __RX_INCLUDE_H

#include "vg_skin.h"
#include "vg_constants_skin.h"

#define RX_(str)    VGAPPEND(vgRedux_,str)

// Define this to allow slicing via instruction address tracking */
//#define RX_SLICING 1 

#define IS_ALIGNED2_ADDR(aaa_p) (0 == (((UInt)(aaa_p)) & 1))


/*------------------------------------------------------------*/
/*--- Types, etc.                                          ---*/
/*------------------------------------------------------------*/

typedef 
   enum {
      // Special ones;  must be 0..N.
      // IF A NEW ONE IS ADDED, init_special_nodes() MUST BE UPDATED!
      SpError=0,              // should never occur; sentinel for accidental 0
      SpStartup=1,            // static data (used by undelazified pointers)
      SpHeap=2, SpZeroHeap=3, // heap allocated
      SpBrk=4,                // brk()'d
      SpUnknownMem=5,         // uninited memory value (indicates prog. bug)
      SpUnknownReg=6,         // uninited register value (also prog. bug?)
      SpSyscallReg=7,         // reg clobbered by syscall retval (should
                              // always be overwritten by skin)
      SpSignalReg=8,          // register clobbered by signal delivery
      SpPthOpReg=9,           // XXX: should never be seen?
      SpClReqReg=10,          // reg clobbered by client request retval
      SpRep=11,               // ECX/EDI/ESI changed by a REP instruction
      SpEspEbp=12,            // Boring %esp/%ebp calculation
      SpLast=13,              // Not used; all special ones must come before it

      // Non-special ones
      ShConst=20,

      ShAdd=21, ShSub=22, ShMul=23,
      ShMul64High=24, ShMul64=25, ShDiv64=26, ShMod64=27,
      ShAnd=28, ShOr=29, ShXor=30,
      ShShl=31, ShShr=32, ShSar=33, ShRol=34, ShRor=35,
      ShInc=36, ShDec=37,
      ShNot=38, ShNeg=39,

      ShLea1Add=40, ShLea2Add=41,

      ShCond=42, ShCC2Val=43,

      ShWiden=44,

      ShShrd=45, ShShld=46,

      ShSyscall=47, ShSysMem=48,

      ShSplit4B=49, ShSplit2W=50, ShSplit2B=51, ShReadB=52, ShReadW=53,

      ShExit=54,

      ShFn=55, ShFnOut=56,

      ShChunk=57, ShString=58,

      ShInd=59,

      ShAlloc=60,

      ShLast                  // Not used; all others must come before it
   }
   ShKind;

typedef
   enum {
      Sz0=0, Sz1=1, Sz2=2, Sz4=4,
   } 
   RxSize;

typedef
   enum {
      ConstError=0, ConstStartup=1, ConstCode=2, /*ConstMmap,*/ ConstWiden=3,
      ConstLea1=4, ConstLea2=5, ConstZero=6, 
      ConstSpecial=7, /*RDTSC*/
      ConstLast   // this must be the last one
   }
   ConstOrigin;

typedef
   enum {
      AllocError=0, AllocMalloc=1, AllocNew=2, AllocVecNew=3, AllocMemAlign=4,
      AllocCalloc=5, AllocRealloc=6,
      AllocLast   // this must be the last one
   }
   AllocKind;

typedef
   enum {
      PZero=0, POne=1, PMany=2
   }
   PCount;

typedef
   struct _ShW {
       // word 1
       UInt   kind:8;                  // node kind
       UInt   extra:8;                 // extra info for some node kinds
       UInt   argc:8;                  // Num of args for variable sized ones
       RxSize size:3;                  // Sz0, Sz1, Sz2 or Sz4;  3 bits used
                                       //   so that Sz4 can be 4 (a bit lazy)

       PCount n_parents:2;             // number of dependents (PZero..PMany)
       UInt   rewritten:1;             // has node been rewritten?
       UInt   graphed:1;               // has node been graphed?

       // word 2
       UInt   val;                     // actual value represented by node

#ifdef RX_SLICING
       // word 3
       Addr   instr_addr;              // address of original x86 instruction
#endif
       // words 3+
       struct _ShW* argv[0];           // variable number of args;
   }
   ShW;

/*--------------------------------------------------------------------*/
typedef
   enum { RxDot, RxAiSee } 
   RxFormat;

/*--------------------------------------------------------------------*/
extern ShW*  RX_(specials)[];    // The special nodes

/*--------------------------------------------------------------------*/
typedef
   struct {
      Char*  name;
      UInt   argc;
      Char** argv;            // names of args
      UInt   min_mem_reads;   // min number of memory reads done
      UInt   max_mem_reads;   // max number of memory reads done
   } 
   SyscallInfo;

#define MAX_NUM_SYSCALLS      256   // Highest seen in vg_syscalls.c: 237

/* Sparsely filled array;  index == syscallno;  defined in rx_main.c */
extern SyscallInfo RX_(syscall_info) [MAX_NUM_SYSCALLS];

/*--------------------------------------------------------------------*/
typedef
   struct _SyscallMade {
      struct _SyscallMade* next;
      ShW*  node;
   }
   SyscallMade;

extern SyscallMade* RX_(syscalls_made);
extern SyscallMade* RX_(syscalls_made_last);

/*------------------------------------------------------------*/
/*--- Command line options + defaults                      ---*/
/*------------------------------------------------------------*/

/* User options */

/* Default print depth?  default: 10 */
extern Int  RX_(clo_print_depth);

/* Ignore %esp/%ebp computations?   default: YES */
extern Bool RX_(clo_ignore_esp_ebp);

/* Show conditional jump nodes?  default: NO */
extern Bool RX_(clo_jcc);

/* Skip pre-main() stuff?  default: NO */
extern Bool RX_(clo_skip);

/* Inline constants in child nodes?  default: 2 (all) */
extern UInt RX_(clo_inline);

/* Rewrite graph to be smaller and prettier?  default: YES */
extern Bool RX_(clo_rewrite);

/* Fit graph onto an a4 sheet? (dot output only)  default: NO */
extern Bool RX_(clo_a4);

/* Output format?  default: dot */
extern RxFormat RX_(clo_format);

/* Debug options */

/* Debug instrumentation phase?  default: NO */
extern Bool RX_(clo_instr);

/* Debug execution phase?  default: NO */
extern Bool RX_(clo_actions);

/* Extra sanity checking?  default: NO */
extern Int  RX_(clo_sanity);

/* Debug BB entry?  default: NO */
extern Bool RX_(clo_bb_entry);

/* After splitting a node for a get, set the shadow as split?  
   (if everything is right, should make no difference to the result)
   default: yes */
extern Bool RX_(clo_write_splits);


/*------------------------------------------------------------*/
/*--- Functions                                            ---*/
/*------------------------------------------------------------*/

/* Functions defined in rx_main.c */
extern Bool RX_(is_special_node) ( ShKind kind );

extern __attribute__((regparm (1)))
ShW* RX_(get_mem_shadow) ( Addr a );

extern Bool  RX_(sane_ShW)        ( ShW* sh, Bool building );

/* Functions defined in rx_print.c */
extern void RX_(pp_ShW)   ( ShW* sh );
extern void RX_(up_ShW)   ( ShW* sh );
extern void RX_(ppp_ShW)  ( Char* str, ShW* sh );

extern void RX_(gr_graph) ( UInt conds_n, ShW* conds_made[] );
extern void RX_(gr_var)   ( Addr a );

extern void RX_(dmp_ShW)  ( ShW* sh );
extern void RX_(dmp_var)  ( Addr a  );

#endif

/*--------------------------------------------------------------------*/
/*--- end                                             rx_include.h ---*/
/*--------------------------------------------------------------------*/

