/***************************************************************************
 *
 * Copyright (c) 1999 Balzs Scheidler
 * Copyright (c) 1999 BalaBit Computing
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Inspired by nsyslog, originally written by Darren Reed.
 *
 * $Id: main.c,v 1.21 1999/11/22 18:26:16 bazsi Exp $
 *
 ***************************************************************************/

#include "syslog-ng.h"
#include "cfgfile.h"
#include "afinter.h"
#include "format.h"
#include "werror.h"

#include <sys/types.h>
#include <signal.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <getopt.h>
#include <wait.h>

#include "main.c.x"

char cfgfilename[128] = PATH_SYSLOG_NG_CONF;
char pidfilename[128] = PATH_PIDFILE;

int restarting = 0;

/* CLASS:
   (class
     (name reread_config)
     (super callback)
     (vars
       (backend object syslog_backend)
       (state simple int)))
*/

static void do_reread_config(struct callback *c)
{
	CAST(reread_config, self, c);

	switch (self->state) {
	case 0:
	case 2:
		self->backend->persistent = make_persistent_config();
		CONFIG_DESTROY(self->backend->configuration, self->backend->persistent);
		self->state++;
		self->backend->oldconfig = self->backend->configuration;
		self->backend->configuration = self->backend->newconfig;
		io_callout(&self->backend->super, 0, c);
		break;
	case 1:
		if (CONFIG_INIT(self->backend->configuration, self->backend->persistent)) {
			notice("new configuration initialized\n");
			self->backend->newconfig = NULL;
			self->backend->oldconfig = NULL;
			kill_persistent_config(self->backend->persistent);
			self->backend->persistent = NULL;
			restarting = 0;
		}
		else {
			/* initializing the new configuration was not 
			   successful, try to revert to the old one */
			self->state++;
			self->backend->newconfig = self->backend->oldconfig;
			self->backend->oldconfig = NULL;
			io_callout(&self->backend->super, 0, c);

		}
		break;
	case 3:
		if (CONFIG_INIT(self->backend->configuration, self->backend->persistent)) {
			notice("Initializing new configuration failed, reverting to old config.\n");
		}
		else {
			/* this is a fatal error, but no stderr & no log output */
			exit(137);
		}
		kill_persistent_config(self->backend->persistent);
		self->backend->persistent = NULL;
		restarting = 0;
		break;
	}
}

/* CLASS:
   (class
     (name syslog_backend)
     (super io_backend)
     (vars
       (newconfig object syslog_config)
       (oldconfig object syslog_config)
       (persistent object persistent_config)
       (configuration object syslog_config)))
*/

int sighuprecvd = 0;

void sig_hup(int signo)
{
	sighuprecvd = 1;
	signal(SIGHUP, sig_hup);
}

int sigtermrecvd = 0;

void sig_term(int signo)
{
	sigtermrecvd = 1;
	signal(SIGTERM, sig_term);
}

void sig_child(int signo)
{
	while (waitpid(-1, NULL, WNOHANG) != -1)
		;
	signal(SIGCHLD, sig_child);
}

int main_loop(struct syslog_backend *backend)
{
	int exit_main_loop = 0;

	signal(SIGPIPE, SIG_IGN);
	signal(SIGHUP, sig_hup);
	signal(SIGTERM, sig_term);
	signal(SIGCHLD, sig_child);

	while (io_iter(&backend->super)) {
		if (exit_main_loop)
			break;
		if (sighuprecvd && !restarting) {

			notice("SIGHUP received, restarting syslog-ng\n");
			restarting = 1;
			backend->newconfig = make_syslog_config(cfgfilename, &backend->super);
			if (backend->newconfig) {
				static struct reread_config r = 
				{ { STATIC_HEADER, do_reread_config }, NULL, 0 };
				r.backend = backend;
				r.state = 0;
				io_callout(&backend->super, 0, &r.super);
			}
			else {
				notice("Syntax error reading config file, using old configuration\n");
			}
			sighuprecvd = 0;
		}
		if (sigtermrecvd) {
			notice("syslog-ng version %z going down\n", VERSION);
			CONFIG_DESTROY(backend->configuration, 0);
			exit_main_loop = 1;
		}
	}
	if (!exit_main_loop) {
		notice("No input/output channels opened, exiting...\n");
		CONFIG_DESTROY(backend->configuration, 0);
	}
	return 0;
}

void go_background()
{
	pid_t pid;
	
	pid = fork();
	if (pid == 0) {
		int fd = open(pidfilename, O_CREAT | O_WRONLY | O_NOCTTY | O_TRUNC, 0600);
		if (fd != -1) {
			struct ol_string *pid_s = c_format("%i", getpid());
			write(fd, pid_s->data, pid_s->length);
			ol_string_free(pid_s);
			close(fd);
		}
		return;
	}
	else if (pid == -1) {
		werror("Cannot fork(), (%z)\n", strerror(errno));
		exit(1);
	}
	signal(SIGTERM, sig_term);
	while (sigtermrecvd == 0)
		pause();
	exit(0);
}

void usage(void)
{
	werror("Usage: syslog-ng [options]\n"
	       "Accept and manage system log messages\n\n"
	       "Options:\n"
	       "  -d, --debug                      Turn on debugging messages\n"
	       "  -v, --verbose                    Be a bit more verbose\n"
	       "  -f <fname>, --cfgfile=<fname>    Set config file name, default=" PATH_SYSLOG_NG_CONF "\n"
	       "  -p <fname>, --pidfile=<fname>    Set pid file name, default=" PATH_PIDFILE "\n"
#ifdef YYDEBUG
	       "  -y, --yydebug                    Turn on yacc debug messages\n"
#endif
		);

	exit(0);
}

#ifdef YYDEBUG
extern int yydebug;
#endif

int main(int argc, char **argv)
{
	int do_fork = 1;
	int opt;

	NEW(syslog_backend, backend);
	struct option syslog_ng_options[] = {
		{ "cfgfile", required_argument, NULL, 'f' },
		{ "pidfile", required_argument, NULL, 'p' },
		{ "debug", no_argument, NULL, 'd' },
		{ "verbose", no_argument, NULL, 'v' },
		{ "help", no_argument, NULL, 'h' },
#ifdef YYDEBUG
		{ "yydebug", no_argument, NULL, 'y' },
#endif
		{ NULL, 0, NULL, 0 }
	};
#ifdef HAVE_MALLOPT
        mallopt(M_TOP_PAD, 0x10000);
#endif
	gc_threshold = 1000;

	while ((opt = getopt_long(argc, argv, "f:p:dvhy", syslog_ng_options, NULL)) != -1) {
		switch (opt) {
		case 'f':
			strncpy(cfgfilename, optarg, sizeof(cfgfilename));
			break;
		case 'p':
			strncpy(pidfilename, optarg, sizeof(pidfilename));
			break;
		case 'd':
			debug_flag++;
			break;
		case 'v':
			verbose_flag++;
			break;
#ifdef YYDEBUG
		case 'y':
			yydebug = 1;
			break;
#endif
		case 'h':
		default:
			usage();
			break;
		}
	}

	init_backend(&backend->super);
	
	backend->configuration = make_syslog_config(cfgfilename, &backend->super);
	if (!backend->configuration) {
		werror("Parse error reading configuration file, exiting.\n");
		return 1;
	}
	if (debug_flag)
		do_fork = 0;
	if (do_fork)
		go_background();
	if (!CONFIG_INIT(backend->configuration, NULL)) {
		werror("Error initializing configuration, exiting.\n");
		if (do_fork) kill(getppid(), SIGTERM);
		return 2;
	}

	if (do_fork) {
		set_error_internal();
		close(0);
		close(1);
		close(2);
		setsid();
		kill(getppid(), SIGTERM); 
	}
	notice("syslog-ng version %z starting\n", VERSION);
	return main_loop(backend);
}
