/***************************************************************************
 *
 * Copyright (c) 1999 Balzs Scheidler
 * Copyright (c) 1999 BalaBit Computing
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Inspired by nsyslog, originally written by Darren Reed.
 *
 * $Id: sources.c,v 1.17 1999/08/29 17:37:56 bazsi Exp $
 *
 ***************************************************************************/

#include "sources.h"
#include "xalloc.h"
#include "format.h"
#include "utils.h"
#include "cfgfile.h"

#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#define  CLASS_DEFINE
#include "sources.h.x"
#undef CLASS_DEFINE

#include "sources.c.x"

/* CLASS:
     (class
       (name log_reader)
       (super read_handler)
       (vars
         (userecv simple UINT32)
	 (pos simple UINT32)
	 (buffer array UINT8 MAX_LINE)
         (next object log_handler)))
*/

static int do_handle_line(struct log_reader *self, UINT32 length, UINT8 *data, abstract_addr *addr, size_t addrlen)
{
	struct log_info *logmsg;

	logmsg = make_log_info(length, data);
	if (addrlen) {
		logmsg->saddr = sockaddr2address_info(addrlen, addr);
	}
	HANDLE_LOG(self->next, logmsg);
	return ST_OK | ST_GOON;
}

static int do_read_line(struct read_handler **h, struct abstract_read *read)
{
	CAST(log_reader, closure, *h);
	UINT8 *eol;
	UINT32 length;
	int n;
	char sabuf[256];
	size_t salen = sizeof(sabuf);

	if (!closure->userecv) {
		n = A_READ(read, MAX_LINE - closure->pos, closure->buffer + closure->pos);
		salen = 0;
	}
	else
		n = A_RECV(read, MAX_LINE - closure->pos, closure->buffer + closure->pos, (abstract_addr *) &sabuf, &salen);

	switch(n) {
	case 0:
		return ST_OK | ST_GOON;
	case A_FAIL:
		/* Fall throw */
	case A_EOF:
		return ST_FAIL | ST_CLOSE;
	}
	closure->pos += n;

	eol = memchr(closure->buffer, '\0', closure->pos);
	if (eol == NULL)
		eol = memchr(closure->buffer, '\n', closure->pos);
	if (!eol && closure->userecv && closure->pos) {
		/* HP-UX kludge workaround */
		do_handle_line(closure, closure->pos, closure->buffer, salen ? (abstract_addr *) &sabuf : NULL, salen);
		closure->pos = 0;
		return ST_OK | ST_GOON;
	}
	while (eol) {
		/* eol points at the newline character. end points at the
		 * character terminating the line, which may be a carriage
		 * return preceeding the newline. */
		UINT8 *end = eol;

		while ((end > closure->buffer) && (end[-1] == '\r' || end[-1] == '\n' || end[-1] == 0))
			end--;

		length = end - closure->buffer;
		
		do_handle_line(closure, length, closure->buffer, salen ? (abstract_addr *) &sabuf : NULL , salen);

		{
			/* Remove line from buffer */
			/* Number of characters that have been processed */
			UINT32 done = eol - closure->buffer + 1;
			UINT32 left = closure->pos - done;

			memcpy(closure->buffer, closure->buffer + done, left);
			closure->pos = left;
		}
		
		eol = memchr(closure->buffer, '\0', closure->pos);
		if (eol == NULL)
			eol = memchr(closure->buffer, '\n', closure->pos);

	}
	return ST_OK | ST_GOON;
}

struct read_handler *make_log_reader(UINT32 userecv, struct log_handler *next)
{
	NEW(log_reader, self);

	self->super.handler = do_read_line;
	self->userecv = userecv;
	self->next = next;
	
	return &self->super;
}

/* source_group */

static struct ol_string *get_source_hostname(struct address_info *a)
{
	struct ol_string *name;
	static struct ol_string *hostname = NULL;
	
	if (a && a->super.isa == &inet_address_info_class) {
		CAST(inet_address_info, inet_addr, a);
	        struct hostent *hp;
		char *hname, *p;

		hp = gethostbyaddr((char *) &(inet_addr->sa.sin_addr), sizeof(struct in_addr), AF_INET);
		if (!hp) {
			return ol_string_use(inet_addr->ip);
		}
		else {
			hname = hp->h_name;
			p = strchr(hname, '.');
			if (p) *p = 0;
		}
		name = c_format_cstring("%z", hname);
	}
	else {
		if (!hostname) {
			char buf[128];
			hostname = c_format_cstring("%z", getshorthostname(buf, sizeof(buf)));
		}

		ol_string_use(hostname);
		name = hostname;
	}
	return name;
}

static void do_add_source_name(struct log_handler *c, struct log_info *logmsg)
{
	CAST(log_source_group, self, c);
	struct ol_string *name = get_source_hostname(logmsg->saddr);

	assert(self->super.next);
	logmsg->source = c;
	if (self->long_hostnames) {
		if (logmsg->flags & LF_LOCAL) {
			/* local */
			ol_string_free(logmsg->host);
			logmsg->host = c_format("%S@%fS", self->name, name);
		}
		else if (!logmsg->host) {
			/* remote && no hostname */
			logmsg->host = c_format("%S/%fS", name, name);
		} 
		else {
			/* everything else, append source hostname */
			if (logmsg->host)
				logmsg->host = c_format("%fS/%fS", logmsg->host, name);
			else
				logmsg->host = c_format("%fS", name);
		}
	}
	else {
		ol_string_free(logmsg->host);
		logmsg->host = c_format("%fS", name);
	}
	HANDLE_LOG(self->super.next, logmsg);
}

static int do_init_group(struct log_handler *c, struct syslog_config *cfg, struct persistent_config *persistent)
{
	CAST(log_source_group, self, c);
	int res;
	
	struct log_source_driver *drv;
	
	self->long_hostnames = cfg->long_hostnames;
	res = 0;
	for (drv = self->drivers; drv; drv = drv->next_driver) {
		res = LOG_HANDLER_INIT(drv, cfg, persistent);
		if (res & ST_QUIT) 
			break;
	}
	return res;
}

static void do_destroy_group(struct log_handler *c, struct syslog_config *cfg, struct persistent_config *persistent)
{
	CAST(log_source_group, self, c);
	
	struct log_source_driver *drv;
	
	self->long_hostnames = cfg->long_hostnames;
	for (drv = self->drivers; drv; drv = drv->next_driver) {
		if (drv->super.super.destroy)
			LOG_HANDLER_DESTROY(drv, cfg, persistent);
	}
}

void set_source_drivers(struct log_source_group *src, struct log_source_driver *drvs)
{
	struct log_source_driver *drv;
	
	src->drivers = drvs;
	for (drv = src->drivers; drv; drv = drv->next_driver) {
		append_log_handler(drv, src);
	}
}

struct log_source_group *make_source_group(const char *name, struct log_source_driver *drvs)
{
	NEW(log_source_group, self);
  
	self->super.super.init = do_init_group;  
	self->super.super.destroy = do_destroy_group;
	self->super.super.handler = do_add_source_name;
	self->name = c_format("%z", name);
	set_source_drivers(self, drvs);
	
	return self;
}
