/***************************************************************************
 *
 * Copyright (c) 1999 Balzs Scheidler
 * Copyright (c) 1999 BalaBit Computing
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Inspired by nsyslog, originally written by Darren Reed.
 *
 * $Id: cfgfile.c,v 1.22 1999/10/20 19:30:01 bazsi Exp $
 *
 ***************************************************************************/

#include "cfgfile.h"
#include <stdio.h>
#include <string.h>


#define CLASS_DEFINE
#include "cfgfile.h.x"
#undef CLASS_DEFINE

struct syslog_conf *configuration;

void add_dest_group(struct log_dest_group *grp)
{
	grp->next_dest_group = configuration->destinations;
	configuration->destinations = grp;
}

void add_source_group(struct log_source_group *grp)
{
	grp->next_source_group = configuration->sources;
	configuration->sources = grp;
}

void add_filter_rule(struct log_filter *filter)
{
	filter->next_filter = configuration->filters;
	if (filter->next_filter)
	  filter->next_filter->prev_filter = filter;
	filter->prev_filter = NULL;
	configuration->filters = filter;
}

void add_log_connection(struct log_connection *conn)
{
	conn->next = configuration->connections;
	configuration->connections = conn;
}

#define RESOLVE_VAR(s, first, next, i) \
        for (s = first; s; s = s->next) { \
	         if (i->name->length == s->name->length && \
		     strncmp(i->name->data, s->name->data, i->name->length) == 0) { \
			 i->ref = (struct ol_object *) s; \
			 break; \
		 } \
        } \
	if (i->ref == NULL) { \
		 werror("unresolved reference: %S\n", i->name); \
		 return 0; \
	} \


static int do_init_config(struct syslog_conf *self)
{
	struct log_source_group *s;
	struct log_filter *f;
	struct log_dest_group *d;
	struct log_connection *c;
	struct log_endpoint_info *i;
	struct log_handler *center;
	int res;

	center = make_log_center(self->connections);
	
	for (s = self->sources; s; s = s->next_source_group) {
		res = LOG_HANDLER_INIT(s, self);
		append_log_handler(s, center);
		if (res & ST_FAIL)
			return 0;
	}
	
	for (d = self->destinations; d; d = d->next_dest_group) {
		res = LOG_HANDLER_INIT(d, self);
		if (res & ST_FAIL)
			return 0;
	}

	if (!self->internal) {
		werror("Warning: Internal messages are not referenced, they'll go to /dev/null\n");
	}
	
	for (c = self->connections; c; c = c->next) {
		for (i = c->nodes; i; i = i->next) {
			switch (i->type) {
			case EP_SOURCE:
				RESOLVE_VAR(s, self->sources, next_source_group, i);
				break;
			case EP_FILTER:
				RESOLVE_VAR(f, self->filters, next_filter, i);
				break;
			case EP_DESTINATION:
				RESOLVE_VAR(d, self->destinations, next_dest_group, i);
				break;
			}
		}
	}
	return 1;
}

static int do_destroy_config(struct syslog_conf *self)
{
	struct log_dest_group *d;
	struct log_source_group *s;

	for (s = self->sources; s; s = s->next_source_group) {
		if (s->super.super.destroy)
			(void) LOG_HANDLER_DESTROY(s, self);
	}
	
	for (d = self->destinations; d; d = d->next_dest_group) {
		if (d->super.destroy)
			(void) LOG_HANDLER_DESTROY(d, self);
	}
	
	KILL_RESOURCE_LIST(self->resources);
	return 0;
}

extern FILE *yyin;
extern int yyparse();
extern int yydebug;

struct syslog_conf *make_syslog_conf(const char *name, struct io_backend *backend)
{
	FILE *cfg;
	int res;
	
	NEW(syslog_conf, self);
	
	configuration = self;
	self->init = do_init_config;
	self->destroy = do_destroy_config;
	self->backend = backend;
	self->sync_freq = 0;
	self->mark_freq = 1200; /* 20 minutes */
	self->long_hostnames = 1;
	self->time_reopen = 60;
	self->log_fifo_size = 100;
	if ((cfg = fopen(name, "r")) != NULL) {
		yyin = cfg;
		res = yyparse();
		fclose(cfg);
		if (!res) {
			/* successfully parsed */
			self->resources = empty_resource_list();
			return self;
		}
	}
	else {
		werror("Cannot open configuration file %z for reading\n", name);
	}
	KILL(self);
	configuration = NULL;
	return NULL;
}

