/***************************************************************************
 *
 * Copyright (c) 1999 BalaBit Computing
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Inspired by nsyslog, originally written by Darren Reed.
 *
 * $Id: afstreams.c,v 1.2 1999/07/10 15:58:33 bazsi Exp $
 *
 ***************************************************************************/

#include "afstreams.h"
#include "sources.h"
#include "format.h"
#include "cfgfile.h"

#include "afstreams.c.x"


#if HAVE_STRLOG_H && HAVE_STROPTS_H

#include <sys/types.h>
#include <sys/stat.h>
#include <stropts.h>
#include <sys/strlog.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>

#ifdef HAVE_DOOR_H
#include <door.h>
#endif

static void prepare_stream(struct nonblocking_fd *fd)
{
	if (fd->to_be_closed)
		kill_fd(fd);
}

/* CLASS:
   (class
     (name stream_read)
     (super abstract_read)
     (vars
       (fd simple int)))
*/

static int do_stream_read(struct abstract_read **r, UINT32 length, UINT8 *buf)
{
	CAST(stream_read, self, *r);
	struct strbuf ctl, data;
	struct log_ctl lc;
	int flags;
	/* char tmp[MAX_LINE + 1]; */
	int res;

	ctl.maxlen = ctl.len = sizeof(lc);
	ctl.buf = (char *) &lc;
	data.maxlen = sizeof(tmp);
	data.len = 0;
	data.buf = tmp;
	flags = 0;

	res = getmsg(self->fd, &ctl, &data, &flags));
	if ((res & MORECTL) == 0) {
		int reqlen;
		if (res & MOREDATA) {
			werror("do_stream_read: STREAMS device gives too long line\n");
		}
		reqlen = c_format_length("<%d>%s", lc.pri, data.len, data.buf);
		return c_format_write("<%d>%s", length, buf, lc.pri, data.len + (length < reqlen) ? length - reqlen : 0, data.buf);
	}
	else {
		werror("do_stream_read: getmsg() trying to return too much ctl data, res=%i %z\n", res, strerror(errno));
		return ST_OK | ST_GOON;
	}
}

static void stream_read_callback(struct nonblocking_fd *fd)
{
	CAST(io_fd, self, fd);
	struct stream_read r = { { STACK_HEADER, do_stream_read, NULL }, fd->fd };
	int res;
	
        /* The handler function may install a new handler */
        res = READ_HANDLER(self->handler, &r.super);

        if (res & ST_DIE)
        {
                closekill_fd(fd,
                             ST_FAILUREP(res) ? CLOSE_PROTOCOL_FAILURE : 0);
        }
        else if (res & ST_CLOSE)
        {
                close_fd(fd, ST_FAILUREP(res) ? CLOSE_PROTOCOL_FAILURE : CLOSE_EOF);
        }

}

struct io_fd *io_stream_get(struct io_fd *f,
			    struct read_handler *handler,
			    struct close_callback *close)
{
	f->super.prepare = prepare_stream;
	f->super.read = stream_read_callback;
	f->super.want_read = !!handler;

	f->handler = handler;
	f->super.close_callback = close;
	return f;
}

/* CLASS:
   (class
     (name afstreams_source)
     (super log_source_driver)
     (vars
       (name string)
       (door_name string)
       (stream_fd object io_fd)
       (door simple int)))
*/

#ifdef HAVE_DOOR_H
void afstreams_door_server_proc(void *cookie, char *argp, size_t arg_size, door_desc_t *dp, size_t n_desc)
{
        door_return(NULL, 0, NULL, 0);
        return;
}
#endif

void afstreams_set_sundoor(struct log_source_driver *c, const char *door)
{
	CAST(afstreams_source, self, c);
	self->door_name = c_format_cstring("%z", door);
}

static int do_init_afstreams_source(struct log_handler *c, struct syslog_conf *cfg)
{
	CAST(afstreams_source, self, c);
	int fd;

	fd = open(self->name->data, O_RDONLY | O_NOCTTY | O_NONBLOCK);
	if (fd != -1) {
		struct strioctl ioc;
		struct stat st;
		
		memset(&ioc, 0, sizeof(ioc));
		ioc.ic_cmd = I_CONSLOG;
		if (ioctl(fd, I_STR, &ioc) < 0) { 
			werror("do_init_afstreams_source: Cannot enable console logging on sun-stream %S (%z)\n", self->name, strerror(errno));
			close(fd);
			return ST_FAIL | ST_QUIT;
		}
		self->stream_fd = 
			io_stream_get(
				make_io_fd(cfg->backend, fd),
				make_log_reader(0, c),
				NULL);
		REMEMBER_RESOURCE(cfg->resources, &self->stream_fd->super.super);

#ifdef HAVE_DOOR_H
		if (self->door_name) {
			if (stat(self->door_name->data, &st) == -1) {
				fd = creat(self->door_name->data, 0666);
				if (fd == -1) {
					werror("do_init_afstreams_source: cannot create door file %S (%z)\n", self->door_name, strerror(errno));
					return ST_FAIL | ST_QUIT;
				}
				close(fd);
			}
			fdetach(self->door_name->data);
			fd = door_create(afstreams_door_server_proc, NULL, 0);
	                if (fd == -1) {
        	                werror("do_init_afstreams_source: Cannot initialize door server %S (%z)\n", self->name, strerror(errno));
	                        return ST_FAIL | ST_QUIT;
	                }
	                if (fattach(fd, self->door_name->data) == -1) {
	                        werror("do_init_afstreams_source: Cannot attach door to %S (%z)\n", self->name, strerror(errno));
				close(fd);
	                        return ST_FAIL | ST_QUIT;
			}
			self->door = fd;
	
		}
#endif
	}
	else {
		werror("do_init_afstreams_source: Cannot open sun-stream %S (%z)\n", self->name, strerror(errno));
		return ST_FAIL | ST_QUIT;
	}
	return ST_OK | ST_GOON;
}

static void do_handle_afstreams_source(struct log_handler *c, struct log_info *log)
{
	CAST(afstreams_source, self, c);
	log->flags |= LF_LOCAL;
	HANDLE_LOG(self->super.super.next, log);
}

#ifdef HAVE_DOOR_H
static int do_destroy_afstreams_source(struct log_handler *c, struct syslog_conf *cfg)
{
	CAST(afstreams_source, self, c);
	
	if (self->door != -1) {
		door_revoke(self->door);
		close(self->door);
	}
	return ST_OK | ST_GOON;
}
#endif

struct log_source_driver *make_afstreams_source(const char *name)
{
	NEW(afstreams_source, self);

	self->super.super.super.init = do_init_afstreams_source;
	self->super.super.super.handler = do_handle_afstreams_source;
#ifdef HAVE_DOOR_H
	self->super.super.super.destroy = do_destroy_afstreams_source;
#endif
	self->name = c_format_cstring("%z", name);
	self->door = -1;
	return &self->super;
}

#else

void afstreams_set_sundoor(struct log_source_driver *c, const char *door_name)
{
}

static int do_init_afstreams_source(struct log_handler *c, struct syslog_conf *cfg)
{
	CAST(afstreams_source, self, c);
	werror("sun-streams support not compiled in (STREAM=%S)\n", self->name);
	return ST_FAIL | ST_QUIT;
}

struct log_source_driver *make_afstreams_source(const char *name)
{
	NEW(afstreams_source, self);

	self->super.super.super.init = do_init_afstreams_source;
	self->name = c_format_cstring("%z", name);
	return &self->super;
}


#endif
