/*
 * The Sleuth Kit
 *
 * $Date: 2005/06/13 19:27:25 $
 *
 * Brian Carrier [carrier@sleuthkit.org]
 * Copyright (c) 2003-2005 Brian Carrier.  All rights reserved
 *
 * mmls - list media management structure contents
 *
 * This file is part of mmtools
 *
 * mmtools is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * mmtools is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with mactime; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "mm_tools.h"


extern char *progname;

FILE *logfp;

static uint8_t print_bytes = 0;
static uint8_t recurse = 0;

static int recurse_cnt = 0;
static DADDR_T recurse_list[64];

void
usage()
{
    fprintf(stderr,
	    "%s [-i imgtype] [-o imgoffset] [-brvV] [-t mmtype] image [images]\n",
	    progname);
    fprintf(stderr, "\t-t mmtype: The type of partition system\n");
    fprintf(stderr, "\t-i imgtype: The format of the image file\n");
    fprintf(stderr,
	    "\t-o imgoffset: Offset to the start of the volume that contains the partition system (in sectors)\n");
    fprintf(stderr, "\t-b: print the rounded length in bytes\n");
    fprintf(stderr,
	    "\t-r: recurse and look for other partition tables in partitions (DOS Only)\n");
    fprintf(stderr, "\t-v: verbose output\n");
    fprintf(stderr, "\t-V: print the version\n");
    fprintf(stderr, "Supported partition systems:\n");
    mm_print_types(stderr);
    fprintf(stderr, "Supported image formats:\n");
    img_print_types(stderr);
    exit(1);
}

/*
 * The callback action for the part_walk
 * */
uint8_t
part_act(MM_INFO * mm, PNUM_T pnum, MM_PART * part, int flag, char *ptr)
{
    /* Neither table or slot were given */
    if ((part->table_num == -1) && (part->slot_num == -1))
	printf("%.2" PRIuPNUM ":  -----   ", pnum);

    /* Table was not given, but slot was */
    else if ((part->table_num == -1) && (part->slot_num != -1))
	printf("%.2" PRIuPNUM ":  %.2" PRIuPNUM "      ",
	       pnum, part->slot_num);

    /* The Table was given, but slot wasn't */
    else if ((part->table_num != -1) && (part->slot_num == -1))
	printf("%.2" PRIuPNUM ":  -----   ", pnum);

    /* Both table and slot were given */
    else if ((part->table_num != -1) && (part->slot_num != -1))
	printf("%.2" PRIuPNUM ":  %.2d:%.2d   ",
	       pnum, part->table_num, part->slot_num);

    if (print_bytes) {
	OFF_T size;
	char unit = ' ';
	size = part->len;

	if (part->len < 2) {
	    size = 512 * part->len;
	    unit = 'B';
	}
	else if (size < (2 << 10)) {
	    size = part->len / 2;
	    unit = 'K';
	}
	else if (size < (2 << 20)) {
	    size = part->len / (2 << 10);
	    unit = 'M';
	}
	else if (size < ((OFF_T) 2 << 30)) {
	    size = part->len / (2 << 20);
	    unit = 'G';
	}
	else if (size < ((OFF_T) 2 << 40)) {
	    size = part->len / (2 << 30);
	    unit = 'T';
	}

	/* Print the layout */
	printf("%.10" PRIuDADDR "   %.10" PRIuDADDR "   %.10" PRIuDADDR
	       "   %.4" PRIuOFF "%c   %s\n", part->start,
	       (part->start + part->len - 1), part->len, size, unit,
	       part->desc);
    }
    else {

	/* Print the layout */
	printf("%.10" PRIuDADDR "   %.10" PRIuDADDR "   %.10" PRIuDADDR
	       "   %s\n", part->start, (part->start + part->len - 1),
	       part->len, part->desc);
    }

    if ((recurse) && (mm->mmtype == MM_DOS) && (part->type == MM_TYPE_VOL)) {
	if (recurse_cnt < 64)
	    recurse_list[recurse_cnt++] = part->start;
    }

    return WALK_CONT;
}

static void
print_header(MM_INFO * mm)
{
    printf("%s\n", mm->str_type);
    printf("Sector: %" PRIuDADDR "\n", mm->sect_offset);
    printf("Units are in %d-byte sectors\n\n", mm->block_size);
    if (print_bytes)
	printf
	    ("     Slot    Start        End          Length       Size    Description\n");
    else
	printf
	    ("     Slot    Start        End          Length       Description\n");
}


int
main(int argc, char **argv)
{
    MM_INFO *mm;
    char *mmtype = NULL;
    int ch;
    DADDR_T sect_offset = 0;
    uint8_t flags = 0;
    char *imgtype = NULL;
    IMG_INFO *img;
    logfp = stderr;

    progname = argv[0];

    while ((ch = getopt(argc, argv, "bi:o:rt:vV")) > 0) {
	char *cp;
	switch (ch) {
	case 'b':
	    print_bytes = 1;
	    break;
	case 'i':
	    imgtype = optarg;
	    break;

	    // @@@ WE may want this to support the @ syntax...
	case 'o':
	    sect_offset = strtoull(optarg, &cp, 0);
	    if (*cp || cp == optarg) {
		fprintf(stderr, "Bad sector offset\n");
		usage();
	    }

	    break;
	case 'r':
	    recurse = 1;
	    break;
	case 't':
	    mmtype = optarg;
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    print_version(stdout);
	    exit(0);
	case '?':
	default:
	    fprintf(stderr, "Unknown argument\n");
	    usage();
	}
    }

    /* We need at least one more argument */
    if (optind >= argc) {
	fprintf(stderr, "Missing image name\n");
	usage();
    }

    /* open the image - do not use an offset */
    img =
	img_open(imgtype, "0", argc - optind,
		 (const char **) &argv[optind]);

    /* process the partition tables */
    mm = mm_open(img, mmtype, sect_offset);
    if (mm == NULL) {
	if (mmtype == NULL)
	    printf("Error determining partition system type\n");
	else
	    fprintf(stderr, "Error processing image\n");
	exit(1);
    }

    print_header(mm);

    mm->part_walk(mm, mm->first_part, mm->last_part, flags,
		  part_act, NULL);

    mm->close(mm);
    if ((recurse) && (mm->mmtype == MM_DOS)) {
	int i;
	/* disable recursing incase we hit another DOS partition
	 * future versions may support more layers */
	recurse = 0;

	for (i = 0; i < recurse_cnt; i++) {
	    mm = mm_open(img, NULL, recurse_list[i]);
	    if (mm != NULL) {
		printf("\n\n");
		print_header(mm);
		mm->part_walk(mm, mm->first_part, mm->last_part, flags,
			      part_act, NULL);
		mm->close(mm);
	    }
	}
    }

    img->close(img);
    exit(0);
}
