/*
** ifind (inode find)
** The Sleuth Kit
**
** $Date: 2005/07/08 17:20:10 $
**
** Given an image  and block number, identify which inode it is used by
** 
** Brian Carrier [carrier@sleuthkit.org]
** Copyright (c) 2003-2005 Brian Carrier.  All rights reserved
**
** TASK
** Copyright (c) 2002 Brian Carrier, @stake Inc.  All rights reserved
**
** TCTUTILs
** Brian Carrier [carrier@cerias.purdue.edu]
** Copyright (c) 2001 Brian Carrier.  All rights reserved
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**
** 1. Redistributions of source code must retain the above copyright notice,
**    this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 3. The name of the author may not be used to endorse or promote
**    products derived from this software without specific prior written
**    permission.     
**
**
** THIS SOFTWARE IS NOT AFFILIATED WITH PURDUE UNIVERSITY OR THE CENTER FOR
** EDUCATION IN INFORMATION ASSURANCE AND SECURITY (CERIAS) AND THEY BEAR
** NO RESPONSIBILITY FOR ITS USE OR MISUSE.
**
**
** THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
** WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
**
** IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
** INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
** (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
** BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
** OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
** ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*/

#include "fs_lib.h"



static uint8_t localflags;
static uint8_t found;


/*******************************************************************************
 * Find an NTFS MFT entry based on its parent directory
 */

static FS_DENT *fs_dent = NULL;
static INUM_T parinode = 0;

/* dent call back for finding unallocated files based on parent directory
 */
static uint8_t
ifind_par_act(FS_INFO * fs, FS_INODE * fs_inode, int flags, void *ptr)
{
    FS_NAME *fs_name;

    /* go through each file name structure */
    fs_name = fs_inode->name;
    while (fs_name) {
	if (fs_name->par_inode == parinode) {
	    /* Fill in teh basics of the fs_dent entry */
	    fs_dent->fsi = fs_inode;
	    fs_dent->inode = fs_inode->addr;
	    strncpy(fs_dent->name, fs_name->name, fs_dent->maxnamlen);
	    if (localflags & IFIND_PAR_LONG) {
		fs_dent_print_long(stdout, fs_dent, FS_FLAG_NAME_UNALLOC,
				   fs, NULL);
	    }
	    else {
		fs_dent_print(stdout, fs_dent, FS_FLAG_NAME_UNALLOC, fs,
			      NULL);
		printf("\n");
	    }
	    fs_dent->fsi = NULL;
	    found = 1;
	}
	fs_name = fs_name->next;
    }

    return WALK_CONT;
}




uint8_t
fs_ifind_par(FS_INFO * fs, uint8_t lclflags, INUM_T par)
{
    found = 0;
    localflags = lclflags;
    parinode = par;
    fs_dent = fs_dent_alloc(256);

    fs->inode_walk(fs, fs->first_inum, fs->last_inum,
		   FS_FLAG_META_LINK | FS_FLAG_META_UNLINK |
		   FS_FLAG_META_UNALLOC | FS_FLAG_META_USED,
		   ifind_par_act, NULL);

    fs_dent_free(fs_dent);

    return 0;
}






/*******************************************************************************
 * Find an inode given a file path
 */

static char *cur_dir;
static char *cur_attr;

/* 
 * dent_walk for finding the inode based on path
 *
 * This is run from the main function and from this function when
 * the needed directory is found
 */
static uint8_t
ifind_path_act(FS_INFO * fs, FS_DENT * fs_dent, int flags, void *ptr)
{

    /* This crashed because cur_dir was null, but I'm not sure how
     * it got that way, so this was added
     */
    if (cur_dir == NULL) {
	fprintf(stderr,
		"cur_dir is null: Please run with '-v' and send output to developers\n");
	return WALK_STOP;
    }

    /* 
     * Check if this is the name that we are currently looking for,
     * as identified in 'cur_dir'
     *
     * All non-matches will return from these checks
     */

    if (((fs->ftype & FSMASK) == EXTxFS_TYPE) ||
	((fs->ftype & FSMASK) == FFS_TYPE)) {
	if (strcmp(fs_dent->name, cur_dir) != 0) {
	    return WALK_CONT;
	}
    }

    /* NTFS gets a case insensitive comparison */
    else if ((fs->ftype & FSMASK) == NTFS_TYPE) {
	if (strcasecmp(fs_dent->name, cur_dir) != 0) {
	    return WALK_CONT;
	}

	/*  ensure we have the right attribute name */
	if (cur_attr != NULL) {
	    int fail = 1;

	    if (fs_dent->fsi) {
		FS_DATA *fs_data;
		fs_data = fs_dent->fsi->attr;

		while ((fs_data) && (fs_data->flags & FS_DATA_INUSE)) {
		    if (strcasecmp(fs_data->name, cur_attr) == 0) {
			fail = 0;
			break;
		    }
		    fs_data = fs_data->next;
		}
	    }
	    if (fail) {
		printf("Attribute name (%s) not found in %s: %" PRIuINUM
		       "\n", cur_attr, cur_dir, fs_dent->inode);

		return WALK_STOP;
	    }
	}
    }
    /* FAT is a special case because there could be the short name
     * in parens - abcdefasdfsdfa (abc..~1.sd)
     */
    else if ((fs->ftype & FSMASK) == FATFS_TYPE) {

	/* try the full match first */
	if (strcasecmp(fs_dent->name, cur_dir) == 0) {

	}
	/* Do a quick 2 char sanity check */
	else if (strncasecmp(fs_dent->name, cur_dir, 2) != 0) {
	    return WALK_CONT;
	}
	/* Check if there is a short name by looking for the
	 * paren at the end */
	else if (fs_dent->name[strlen(fs_dent->name) - 1] == ')') {
	    char *sh_ptr;
	    unsigned int long_len = 0, sh_len = 0;

	    /* Get the beginning of the short name */
	    sh_ptr = strrchr(fs_dent->name, '(');
	    if (sh_ptr == NULL) {
		fprintf(stderr,
			"ifind: error parsing FAT name (no '('): %s\n",
			fs_dent->name);
		return WALK_CONT;
	    }

	    /* Advance to the first letter in the name */
	    sh_ptr++;

	    /* Length of long name - 2 for ' (' */
	    long_len = (uintptr_t) sh_ptr - (uintptr_t) fs_dent->name - 2;

	    /* Length of Short name - 3 for ' (' and ')' */
	    sh_len = strlen(fs_dent->name) - long_len - 3;

	    /* Sanity Check - there should be a space after the lfn */
	    if (fs_dent->name[long_len] != ' ') {
		fprintf(stderr, "ifind: error parsing FAT name: %s\n",
			fs_dent->name);
		return WALK_CONT;
	    }

	    /* Check if the long name has the same length as the target */
	    if (strlen(cur_dir) == long_len) {
		if (strncasecmp(fs_dent->name, cur_dir, long_len) != 0) {
		    return WALK_CONT;
		}
	    }
	    /* check if the short name has the same length */
	    else if (strlen(cur_dir) == sh_len) {
		if (strncasecmp(sh_ptr, cur_dir, sh_len) != 0) {
		    return WALK_CONT;
		}
	    }
	    /* The length is not the same, so just return */
	    else {
		return WALK_CONT;
	    }
	}
	/* No short name at the end, so verify it is the right size */
	else if (strlen(fs_dent->name) < 13) {
	    if (strcasecmp(fs_dent->name, cur_dir) != 0) {
		return WALK_CONT;
	    }
	}
	/* No short name and too long - error */
	else {
	    fprintf(stderr, "ifind: Error parsing FAT name: %s\n",
		    fs_dent->name);
	    return WALK_CONT;
	}
    }

    /* Get the next directory or file name */
    cur_dir = (char *) strtok(NULL, "/");
    cur_attr = NULL;

    if (verbose)
	fprintf(stderr, "Found it (%s), now looking for %s\n",
		fs_dent->name, cur_dir);

    /* That was the last one */
    if (cur_dir == NULL) {
	printf("%" PRIuINUM "\n", fs_dent->inode);
	found = 1;
	return WALK_STOP;
    }

    /* if it is an NTFS image with an ADS in the name, then
     * break it up 
     */
    if (((fs->ftype & FSMASK) == NTFS_TYPE) &&
	((cur_attr = strchr(cur_dir, ':')) != NULL)) {
	*cur_attr = '\0';
	cur_attr++;
    }

    /* it is a directory so we can recurse */
    if ((fs_dent->fsi->mode & FS_INODE_FMT) == FS_INODE_DIR) {

	fs->dent_walk(fs, fs_dent->inode,
		      FS_FLAG_NAME_ALLOC | FS_FLAG_NAME_UNALLOC,
		      ifind_path_act, (char *) 0);
    }

    /* The name was correct, but it was not a directory */
    else {
	printf("Invalid path (%s is a file)\n", fs_dent->name);
    }

    return WALK_STOP;
}


uint8_t
fs_ifind_path(FS_INFO * fs, uint8_t lclflags, char *path)
{
    found = 0;
    localflags = lclflags;

    cur_dir = (char *) strtok(path, "/");
    cur_attr = NULL;

    /* If there is no token, then only a '/' was given */
    if (!cur_dir) {
	printf("%lu\n", (ULONG) fs->root_inum);
	return 0;
    }

    /* If this is NTFS, ensure that we take out the attribute */
    if (((fs->ftype & FSMASK) == NTFS_TYPE) &&
	((cur_attr = strchr(cur_dir, ':')) != NULL)) {
	*cur_attr = '\0';
	cur_attr++;
    }

    if (verbose)
	fprintf(stderr, "Looking for %s\n", cur_dir);

    fs->dent_walk(fs, fs->root_inum,
		  FS_FLAG_NAME_ALLOC | FS_FLAG_NAME_UNALLOC,
		  ifind_path_act, NULL);

    if (0 == found) {
	printf("File not found: %s\n", cur_dir);
	return 1;
    }
    return 0;
}





/*******************************************************************************
 * Find an inode given a data unit
 */

static DADDR_T block = 0;	/* the block to find */
static INUM_T curinode;		/* the inode being analyzed */

static uint32_t curtype;	/* the type currently being analyzed: NTFS */
static uint16_t curid;

/*
 * file_walk action for non-ntfs
 */
static uint8_t
ifind_data_file_act(FS_INFO * fs, DADDR_T addr, char *buf,
		    unsigned int size, int flags, void *ptr)
{
    /* Drop references to block zero (sparse)
     * This becomes an issue with fragments and looking for fragments
     * within the first block.  They will be triggered by sparse 
     * entries, even though the first block can not be allocated
     */
    if (!addr)
	return WALK_CONT;

    if ((block >= addr) &&
	(block < (addr + (size + fs->block_size - 1) / fs->block_size))) {
	printf("%" PRIuINUM "\n", curinode);

	if (!(localflags & IFIND_ALL)) {
	    fs->close(fs);
	    exit(0);
	}
	found = 1;
    }
    return WALK_CONT;
}


/* 
 * file_walk action callback for ntfs  
 *
 */
static uint8_t
ifind_data_file_ntfs_act(FS_INFO * fs, DADDR_T addr, char *buf,
			 unsigned int size, int flags, void *ptr)
{
    if (addr == block) {
	printf("%" PRIuINUM "-%" PRIu32 "-%" PRIu16 "\n", curinode,
	       curtype, curid);

	if (!(localflags & IFIND_ALL)) {
	    fs->close(fs);
	    exit(0);
	}
	found = 1;
    }
    return WALK_CONT;
}



/*
** find_inode
**
** Callback action for inode_walk
*/
static uint8_t
ifind_data_act(FS_INFO * fs, FS_INODE * fs_inode, int flags, void *ptr)
{
    int file_flags = (FS_FLAG_FILE_AONLY | FS_FLAG_FILE_NOABORT);

    /* If the meta data structure is unallocated, then set the recovery flag */
    if (flags & FS_FLAG_META_UNALLOC)
	file_flags |= FS_FLAG_FILE_RECOVER;

    curinode = fs_inode->addr;

    /* NT Specific Stuff: search all ADS */
    if ((fs->ftype & FSMASK) == NTFS_TYPE) {
	FS_DATA *data = fs_inode->attr;

	file_flags |= FS_FLAG_FILE_SLACK;
	while ((data) && (data->flags & FS_DATA_INUSE)) {
	    curtype = data->type;
	    curid = data->id;
	    if (data->flags & FS_DATA_NONRES) {
		fs->file_walk(fs, fs_inode, data->type, data->id,
			      file_flags, ifind_data_file_ntfs_act, ptr);
	    }
	    data = data->next;
	}
	return WALK_CONT;
    }
    else if ((fs->ftype & FSMASK) == FATFS_TYPE) {
	file_flags |= (FS_FLAG_FILE_SLACK | FS_FLAG_FILE_NOID);
	fs->file_walk(fs, fs_inode, 0, 0, file_flags, ifind_data_file_act,
		      ptr);
    }
    /* UNIX do not need the SLACK flag because they use fragments - if the
     * SLACK flag exists then any unused fragments in a block will be 
     * correlated with the incorrect inode
     */
    else {
	file_flags |= (FS_FLAG_FILE_NOID);
	fs->file_walk(fs, fs_inode, 0, 0, file_flags, ifind_data_file_act,
		      ptr);
    }

    return WALK_CONT;
}


/*
 * if the block is a meta data block, then report that, otherwise
 * this is where we say that the inode was not found
 */
static uint8_t
ifind_data_block_act(FS_INFO * fs, DADDR_T addr, char *buf, int flags,
		     void *ptr)
{
    if (flags & FS_FLAG_DATA_META)
	printf("Meta Data\n");
    else
	printf("Inode not found\n");

    return WALK_STOP;
}


uint8_t
fs_ifind_data(FS_INFO * fs, uint8_t lclflags, DADDR_T blk)
{

    found = 0;
    localflags = lclflags;
    block = blk;

    fs->inode_walk(fs, fs->first_inum, fs->last_inum,
		   FS_FLAG_META_LINK | FS_FLAG_META_UNLINK |
		   FS_FLAG_META_ALLOC | FS_FLAG_META_UNALLOC |
		   FS_FLAG_META_USED | FS_FLAG_META_UNUSED,
		   ifind_data_act, NULL);

    /* 
     * If we did not find an inode yet, we call block_walk for the 
     * block to find out the associated flags so we can identify it as
     * a meta data block */
    if (0 == found) {
	fs->block_walk(fs, block, block,
		       FS_FLAG_DATA_UNALLOC | FS_FLAG_DATA_ALLOC |
		       FS_FLAG_DATA_META | FS_FLAG_DATA_CONT,
		       ifind_data_block_act, NULL);
    }
    if (0 == found) {
	printf("Inode not found\n");
	return 1;
    }
    return 0;
}
