/*
 * Copyright (C) 1999 Bell Labs, Lucent Technologies.
 * Copyright (C) Arash Baratloo, Timothy Tsai, and Navjot Singh.
 *
 * This file is part of the Libsafe library.
 * Libsafe version 1.x: protecting against stack smashing attacks.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 *
 * For more information, 
 *   visit http://www.bell-labs.com/org/11356/libsafe.html
 *   or email libsafe@research.bell-labs.com
 */

#include <unistd.h>		/* defines readlink() */
#include <syslog.h>		/* defines syslog(3) */
#include <stdarg.h>		/* defines va_args */
#include <signal.h>		/* defines kill() */
#include <stdio.h>
#include <unistd.h>             /* defines pipe() */
#include "util.h"
#include "log.h"

#ifdef NOTIFY_WITH_EMAIL
#include <stdlib.h>
#include <sys/wait.h>
#include <ctype.h>
#define __USE_GNU 1		/* defines strnlen() */
#include <string.h>
#include <stdlib.h>
#endif				/* NOTIFY_WITH_EMAIL */

extern char **environ[];
static void *stack_start = NULL;

void _libsafe_initialize_start_stack() 
{
        stack_start = __builtin_frame_address(2);
}

static char *get_exename(char *exename, int size)
{
	int res;
        
	/*
         * get the name of the current executable
         */
        if ((res = readlink("/proc/self/exe", exename, size - 1)) == -1)
		exename[0] = '\0';
	else
		exename[res] = '\0';
	return (exename);
}

/* Given an address 'addr' returns 0 iff the address does not point to
 * a stack variable.  Otherwise, it returns a positive number
 * indicating the number of bytes (distance) between the 'addr' and
 * the frame pointer it resides in.  Note: stack grows down, and
 * arrays/structures grow up.
 */
uint _libsafe_stackVariableP(void *addr)
{
	/*
	 * (Vandoorselaere Yoann)
	 * We have now just one cast.
	 */
	void *fp, *sp;
        
	/*
	 * (Arash Baratloo / Yoann Vandoorselaere)
	 * use the stack address of the first declared variable
	 * to get the 'sp' address in a portable way.
	 */
	sp = &fp;

        /*
	 * Stack grows downwards (toward 0x00).  Thus, if the stack pointer
	 * is above (>) 'addr', 'addr' can't be on the stack.
         */
        if (sp > addr)
		return 0;

        /*
         * Note: the program name is always stored at 0xbfffffb (documented
         * in the book Linux Kernel).  Search back through the frames to
         * find the frame containing 'addr'.
         */
	fp = __builtin_frame_address(0);

	while ((sp < fp) && (fp <= stack_start)) {
		if (fp > addr)
			return (fp - addr);	/* found the frame */

		fp = *(void **) fp;

	}
	/*
         * reached end of stack and did not find the appropriate frame.
         */
        return 0;
}


#ifdef NOTIFY_WITH_EMAIL
/*
 * Send email to the recipient, with the message as part of the subject.
 */
void sendmail(char *recipient, char *exename, char *message)
{
        char command[500];

      	/*
         * Form the mail command to send mail to root.
         */
        snprintf(command, sizeof(command),
		 "\"libsafe violation for %s, pid=%d; %s\"",
                 exename, getpid(), message);
	/*
         * Echo the mail command to stderr and send out the mail.
         */
	LOG(3, "%s\n", command);

	/*
	 * (Vandoorselaere Yoann)
	 * don't use system cause it can cause security problem,
	 * more importantly, libsafe can be used with suid / sgid
	 * programs.
	 *
	 * from the system() man page :
	 * Do not use system() from a program with suid or sgid priv
	 * ileges,  because strange values for some environment vari
	 * ables might be used to subvert system integrity.  Use  the
	 * exec(3)  family of functions instead, but not execlp(3) or
	 * execvp(3).  system() will not, in fact, work properly from
	 * programs  with suid or sgid privileges on systems on which
	 * /bin/sh is bash version 2, since bash 2  drops  privileges
	 * on  startup.
	 */
	execlp("/bin/mail", "mail", "-s", command, recipient, NULL);
}
#endif


/*
 * This is what is called when a buffer overflow on the stack is
 * detected.  If you want to add customized actions triggered by
 * detection put them here.
 * 'name' is the name of this library, and (format,...) is
 * similar to printf() and passed to syslog().
 */
void _libsafe_die(char *format, ...)
{
        char exename[MAXPATHLEN];
	va_list args;
#ifdef NOTIFY_WITH_EMAIL
	int child_pid;
	int comm_pipe[2];
#endif				/* NOTIFY_WITH_EMAIL */

	/*
         * get the name of the current executable
         */
	get_exename(exename, MAXPATHLEN);
        
	/*
         * add an entry to syslog()
         */
	va_start(args, format);
	openlog(LIBNAME, LOG_CONS | LOG_PID, LOG_AUTHPRIV);
	syslog(LOG_CRIT, "%s", VERSION);
	syslog(LOG_CRIT, "detected an attempt to write across stack boundary.");
	syslog(LOG_CRIT, "terminating %s", exename);
	syslog(LOG_CRIT, format, args);
	closelog();
	va_end(args);

	LOG(1, "Detected an attempt to write across stack boundary.\n");
	LOG(1, "Terminating %s.\n", exename);

	/*
         * PUT ANY CUSTOMIZED ACTIONS HERE...
         */
#ifdef NOTIFY_WITH_EMAIL

        if (pipe(comm_pipe) < 0) {
	        perror("pipe():");
		goto last_resort;
	}
	  
	child_pid = fork();
	if (child_pid == 0) {
		char function[500];	/* Name of the offending function */
		char mail_recipient[500];
		FILE *fp;

		/*
		 * set child's stdin to be one end of the pipe
		 */
	        close(comm_pipe[1]);
		if (dup2(comm_pipe[0], STDIN_FILENO) != STDIN_FILENO) {
		        perror("dup2():");
			_exit(1);
		}
		close(comm_pipe[0]);

                /*
		 * We need this process to run in a separate session, since otherwise the
		 * kill(0,9) command executed by the parent process will kill this child
		 * process.
                 */
		setsid();

                /*
                 * Form the descriptive message.  The message usually indicates the
		 * offending function.
                 */
		va_start(args, format);
		vsnprintf(function, sizeof(function), format, args);
		va_end(args);

                /*
                 * If the mail_list file exists, then send email to all the recipients
                 * listed in that file.  Otherwise, send email to root@localhost.
                 */
		if ((fp = fopen("/etc/libsafe.notify", "r")) == NULL) {
                        sendmail("root@localhost", exename, function);
		} else {
			while (fgets(mail_recipient, sizeof(mail_recipient), fp)) {
				char *p;
                                
                                /*
                                 * Chop off any trailing newlines if present
                                 */
				for (p = mail_recipient + strnlen(mail_recipient, sizeof(mail_recipient)) - 1;
                                     isspace(*p); p--) {
					*p = (char) NULL;
				}
                                
				sendmail(mail_recipient, exename, function);
			}
			fclose(fp);
		}
	} else {
	        close(comm_pipe[0]);
		close(comm_pipe[1]);
	        waitpid(child_pid, NULL, 0);
	}

 last_resort:
                
#endif				/* NOTIFY_WITH_EMAIL */
            /*
             * (Vandoorselaere Yoann)
             * let's do that in a cleaner way,
             * don't use code to generate sigsegv cause it can
             * be handled... use _exit().
             *
             * *(int *)0 = 0;
             */
        _exit(1);
}


