/* StegFS - The Steganographic File System */
/* Copyright (c)1999-2001 Andrew McDonald <andrew@mcdonald.org.uk> */

/* Manipulates the keys section of the btab file. */
/* $Id: stegfsctrl.c,v 1.3 2001/01/30 22:19:52 admcd Exp $ */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <libgen.h>
#include "stegfs.h"
#include <sys/stat.h>
#include <sys/sysmacros.h>
#include <linux/major.h>
#include <string.h>
#include "sha1.h"

#define SC_CMD_INFO	0
#define SC_CMD_ADD	1
#define SC_CMD_REMOVE	2
#define SC_CMD_INIT	3

void printusage(char *);
int verify_skeys(struct stegfs_fs *fs, int level);

int main(int argc, char* argv[]) {

	struct stegfs_fs *fs;
	int i,j,c;
	char *passwds[STEGFS_MAX_LEVELS-1];
	FILE *devrand;
	struct stat statbuf;
	int pos;

	char cipherstr[35];
	int ic, bc;
	char promptbuf[80];
	char *pass;
	char enckey[STEGFS_KEY_BYTES];
	int force;
	int context, context2, level;
	int command;

	fs = (struct stegfs_fs *) malloc(sizeof(struct stegfs_fs));

	printf(STEGFSVERSTR "\n");


	if (argc < 4)
		printusage(argv[0]);


	/* defaults */
	ic = 5;
	bc = 5;
	strncpy(cipherstr, "aes", 35);
	force = 0;

	while (1) {

		c = getopt(argc, argv, "c:i:b:f");

		if (c == -1)
			break;

		switch (c)
		{
		case 'c':
			strncpy(cipherstr, optarg, 35);
			cipherstr[34] = '\0';
			break;
		case 'i':
			ic = atoi(optarg);
			break;
		case 'b':
			bc = atoi(optarg);
			break;
		case 'f':
			force = 1;
			break;

		default:
			printusage(argv[0]);
		}
	}

/*
	if(optind != argc-4) {
		printf("Something not specified\n");
		printusage(argv[0]);
	}
*/
	if (strcmp(argv[optind+2], "info") == 0)
		command = SC_CMD_INFO;
	else if (strcmp(argv[optind+2], "add") == 0)
		command = SC_CMD_ADD;
	else if(strcmp(argv[optind+2], "remove") == 0)
		command = SC_CMD_REMOVE;
	else if(strcmp(argv[optind+2], "init") == 0)
		command = SC_CMD_INIT;
	else
		printusage(argv[0]);

	/* validate args */
	if (ic < 1 || ic > STEGFS_MAX_INO_COPIES) {
		printf("Inode copies: invalid value\n");
		exit(1);
	}
	if (bc < 1 || bc > STEGFS_MAX_BLOCK_COPIES) {
		printf("Block copies: invalid value\n");
		exit(1);
	}

	/* Check random number source is what we think it is */
	i = stat(RAND_DEV, &statbuf);
	if (i == -1) {
		printf("Error accessing %s\n", RAND_DEV);
		exit(3);
	}
	if (!S_ISCHR(statbuf.st_mode) ||
	    major(statbuf.st_rdev) != MEM_MAJOR ||
	    minor(statbuf.st_rdev) != 9) {
		printf("WARNING!!! %s is not char device major %u minor 9!\n",
		       RAND_DEV, MEM_MAJOR);
	}

	if (!open_fs(fs, argv[optind], argv[optind+1], cipherstr, 0)) {
		printf("Failed to open device or file\n");
		exit(2);
	}

	if (command == SC_CMD_INFO) {
		if ((argc-optind) != 3)
			printusage(argv[0]);

		printf("Block table file version: %d\n", fs->btabver);
	}
	else if(command == SC_CMD_ADD) {
		if ((argc-optind) < 5 || (argc-optind) > 6)
			printusage(argv[0]);
		context = atoi(argv[optind+3]);
		if (context < 1 || context > STEGFS_MAX_CONTEXTS) {
			printf("Invalid context number.\n");
			exit(1);
		}
		level = atoi(argv[optind+4]);
		if (level < 0 || level > STEGFS_MAX_LEVELS-1) {
			printf("Invalid level number.\n");
			exit(1);
		}
		if ((argc-optind) == 6) {
			context2 = atoi(argv[optind+5]);
			if (context2 < 1 || context2 > STEGFS_MAX_CONTEXTS) {
				printf("Invalid context number.\n");
				exit(1);
			}
		}
		else
			context2 = 0;
		if (fs->btabver == 1 && level > context && level > context2) {
			printf("Invalid level number with version 1"
			       " block table file.\n");
			exit(1);
		}

		/* get passphrase for context we are altering*/
		sprintf(promptbuf, "Enter passphrase for"
			" security context %u: ", context);
		pass = getpass(promptbuf);
		if(pass[0] == '\0') {
			printf("Aborted.\n");
			exit(1);
		}
		passwds[context-1] = malloc(strlen(pass)+1);
		strncpy(passwds[context-1], pass, strlen(pass)+1);
		memset(pass, 0, strlen(pass));

		/* Get keys */
		get_skeys(fs, context, passwds[context-1]);
		if(!verify_skeys(fs, context)) {
			printf("Invalid passphrase.\n");
			exit(1);
		}

		if(verify_skey(fs, context, level)) {
			printf("     **********\n");
			printf("WARNING: This operation cannot be undone.\n");
			printf("If security context %d is the only context ",
			       context);
			printf("that includes this level\n");
			printf("its contents will be permanently lost.\n");
			printf("     **********\n");

			printf("Are you sure you want to proceed? (y,n) ");
			if (getchar() != 'y') {
				printf("Aborted.\n");
				exit(1);
			}
		}
		if (context2) {
			/* get passphrase for existing context */
			sprintf(promptbuf, "Enter passphrase for"
				" security context %u: ", context2);
			pass = getpass(promptbuf);
			if(pass[0] == '\0') {
				printf("Aborted.\n");
				exit(1);
			}
			passwds[context2-1] = malloc(strlen(pass)+1);
			strncpy(passwds[context2-1], pass, strlen(pass)+1);
			memset(pass, 0, strlen(pass));

			/* Get keys */
			get_skeys(fs, context2, passwds[context2-1]);
			if(!verify_skey(fs, context, level)) {
				printf("Invalid passphrase.\n");
				exit(1);
			}
		}

		if (fs->btabver == 1) {
			pos = fs->e2fs->super->s_blocks_count
				* sizeof(struct stegfs_btable);
			pos += 0.5*context*(context-1)*STEGFS_KEY_BYTES;
		}
		else if (fs->btabver == 2) {
			pos = sizeof(struct stegfs_btable_head) +
				fs->e2fs->super->s_blocks_count
				* sizeof(struct stegfs_btable);
			pos += (context-1)*(STEGFS_MAX_LEVELS-1)*STEGFS_KEY_BYTES;
		}
		else {
			printf("Unknown btab version %u\n", fs->btabver);
			pos = 0;
		}

		pos += (level-1)*STEGFS_KEY_BYTES;
		fseek(fs->btabfile, pos, SEEK_SET);

		if (!context2) {
			devrand = fopen(RAND_DEV, "r");
			if(devrand==NULL) {
				printf("Error opening %s\n", RAND_DEV);
				exit(3);
			}
			fread(fs->slkeys[context-1], 1, STEGFS_KEY_BYTES,
			      devrand);
			fclose(devrand);
		}			

		encrypt_skey(fs, passwds[context-1],
			     fs->slkeys[level-1], enckey);
		fwrite(enckey, sizeof(enckey), 1, fs->btabfile);

		if (!context2) {
			printf("Writing root directory: ");
			fflush(stdout);
			mkroot(fs, context, passwds[context-1], ic, bc);
			printf("Done\n");
		}

	}
	else if (command == SC_CMD_REMOVE) {
		if ((argc-optind) < 4 || (argc-optind) > 5)
			printusage(argv[0]);
		context = atoi(argv[optind+3]);
		if (context < 1 || context > STEGFS_MAX_CONTEXTS) {
			printf("Invalid context number.\n");
			exit(1);
		}
		if ((argc-optind) == 5)
			level = atoi(argv[optind+4]);
		else
			level = 0;
		if (level < 0 || level > STEGFS_MAX_LEVELS-1) {
			printf("Invalid level number.\n");
			exit(1);
		}
		if (fs->btabver == 1 && level > context) {
			printf("Invalid level number with version 1"
			       " block table file.\n");
			exit(1);
		}
		printf("     **********\n");
		printf("WARNING: This operation cannot be undone.\n");
		printf("If security context %d is the only context that ",
		       context);
		if (level)
			printf("includes this level\n");
		else
			printf("includes these levels\n");
		printf("its contents will be permanently lost.\n");
		printf("     **********\n");

		printf("Are you sure you want to proceed? (y,n) ");
		if (getchar() != 'y') {
			printf("Aborted.\n");
			exit(1);
		}

		if (fs->btabver == 1) {
			pos = fs->e2fs->super->s_blocks_count
				* sizeof(struct stegfs_btable);
			pos += 0.5*context*(context-1)*STEGFS_KEY_BYTES;
		}
		else if (fs->btabver == 2) {
			pos = sizeof(struct stegfs_btable_head) +
				fs->e2fs->super->s_blocks_count
				* sizeof(struct stegfs_btable);
			pos += (context-1)*(STEGFS_MAX_LEVELS-1)*STEGFS_KEY_BYTES;
		}
		else {
			printf("Unknown btab version %u\n", fs->btabver);
			pos = 0;
		}

		fseek(fs->btabfile, pos, SEEK_SET);

		devrand = fopen(RAND_DEV, "r");
		if(devrand==NULL) {
			printf("Error opening %s\n", RAND_DEV);
			exit(3);
		}

		if (!level) {
			for (i=0; i<STEGFS_MAX_LEVELS-1; i++) {
				fread(&enckey, 1, STEGFS_KEY_BYTES, devrand);
				fwrite(&enckey, 1, STEGFS_KEY_BYTES,
				       fs->btabfile);
			}
		}		
		else {
			fseek(fs->btabfile,
			      sizeof(struct stegfs_btable)*(level-1),
			      SEEK_CUR);
			fread(&enckey, 1, STEGFS_KEY_BYTES, devrand);
			fwrite(&enckey, 1, STEGFS_KEY_BYTES, fs->btabfile);
		}

		fclose(devrand);
	}
	else if (command == SC_CMD_INIT) {
		if ((argc-optind) != 4)
			printusage(argv[0]);
		context = atoi(argv[optind+3]);
		if (context < 1 || context > STEGFS_MAX_CONTEXTS) {
			printf("Invalid context number.\n");
			exit(1);
		}
		if (context>1) {
			/* get passphrase for existing context */
			sprintf(promptbuf, "Enter passphrase for existing"
				" security context %u: ", context-1);
			pass = getpass(promptbuf);
			if(pass[0] == '\0') {
				printf("Aborted.\n");
				exit(1);
			}
			passwds[context-2] = malloc(strlen(pass)+1);
			strncpy(passwds[context-2], pass, strlen(pass)+1);
			memset(pass, 0, strlen(pass));

			/* Get keys */
			get_skeys(fs, context-1, passwds[context-2]);
			if(!verify_skeys(fs, context-1)) {
				printf("Invalid passphrase.\n");
				exit(1);
			}
		}

		j = 0;

		sprintf(promptbuf, "Enter passphrase for new "
			"security context %u: ", context);
		pass = getpass(promptbuf);
		if(pass[0] == '\0') {
			printf("Aborted.\n");
			exit(1);
		}
		passwds[context-1] = malloc(strlen(pass)+1);
		strncpy(passwds[context-1], pass, strlen(pass)+1);
		memset(pass, 0, strlen(pass));

		sprintf(promptbuf, "Re-enter passphrase for new context %u: ",
			context);
		pass = getpass(promptbuf);
		if (strcmp(pass,passwds[context-1]) != 0) {
			memset(pass, 0, strlen(pass));
			printf("Passphrases differ.\n");
			i--;
			j++;
			if (j > 2) {
				printf("Failed after %d attempts.\n", j);
				exit(1);
			}
		}
		else {
			memset(pass, 0, strlen(pass));
			j = 0;
		}

		printf("     **********\n");
		printf("WARNING: This operation cannot be undone.\n");
		printf("If security context %d already exists its"
		       " set of encrypted level keys\n"
		       "will be permanently lost.\n", context);
		printf("If security context %d is the only context that"
		       " includes any of the levels,\n"
		       "their contents will be permanently lost.\n", context);
		printf("     **********\n");

		printf("Are you sure you want to proceed? (y,n) ");
		if (getchar() != 'y') {
			printf("Aborted.\n");
			exit(1);
		}

		if (fs->btabver == 1) {
			pos = fs->e2fs->super->s_blocks_count
				* sizeof(struct stegfs_btable);
			pos += 0.5*context*(context-1)*STEGFS_KEY_BYTES;
		}
		else if (fs->btabver == 2) {
			pos = sizeof(struct stegfs_btable_head) +
				fs->e2fs->super->s_blocks_count
				* sizeof(struct stegfs_btable);
			pos += (context-1)*(STEGFS_MAX_LEVELS-1)*STEGFS_KEY_BYTES;
		}
		else {
			printf("Unknown btab version %u\n", fs->btabver);
			pos = 0;
		}

		fseek(fs->btabfile, pos, SEEK_SET);

		devrand = fopen(RAND_DEV, "r");
		if(devrand==NULL) {
			printf("Error opening %s\n", RAND_DEV);
			exit(3);
		}

		/* Write new encrypted keys. */
		fs->slkeys[context-1] = malloc(STEGFS_KEY_BYTES);
		fread(fs->slkeys[context-1], 1, STEGFS_KEY_BYTES, devrand);
		for (j=0; j<context; j++) {
			encrypt_skey(fs, passwds[context-1],
				     fs->slkeys[j], enckey);
			fwrite(enckey, sizeof(enckey), 1, fs->btabfile);
		}

		if(fs->btabver > 1) {
			for(j=context; j<STEGFS_MAX_LEVELS-1; j++) {
				fread(enckey, 1, STEGFS_KEY_BYTES, devrand);
				fwrite(enckey, STEGFS_KEY_BYTES, 1,
				       fs->btabfile);
			}
		}

		fclose(devrand);

		printf("Writing root directory: ");
		fflush(stdout);
		mkroot(fs, context, passwds[context-1], ic, bc);
		printf("Done\n");
	}
	else {
		/* Unknown command */
		printusage(argv[0]);
	}

	close_fs(fs);

	return 0;
}




void printusage(char *progname) {
	printf("\nUsage:\n");
	printf("%s /dev/fsdevice btabfile info\n", basename(progname));
	printf("%s /dev/fsdevice btabfile add context level [context] [-c cipher]\n"
	       "           [-i inocopies] [-b blkcopies] [-f]\n",
	       basename(progname));
	printf("%s /dev/fsdevice btabfile remove context level [-c cipher] [-f]\n",
	       basename(progname));
	printf("%s /dev/fsdevice btabfile init context [-c cipher] [-i inocopies]\n"
	       "           [-b blkcopies] [-f]\n",
	       basename(progname));
	/*
	printf("Note: levels_to_add is the number of additional levels to add.\n"
	       "      current_max_level + levels_to_add must be less than or\n"
	       "      equal to %d.\n",
	       STEGFS_MAX_LEVELS-1);
	*/
	exit(1);
}
