#!/usr/bin/perl

#    mkpatch - Create patches against the Linux kernel
#    Copyright (c) 1999  Frodo Looijaard <frodol@dds.nl>
#
#    From the version for lm-sensors
#    Modifications for StegFS
#    Copyright (c) 2001  Andrew McDonald <andrew@mcdonald.org.uk>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

use strict;

use vars qw($temp);
$temp = "mkpatch/.temp";

# Generate a diff between the old kernel file and the new I2C file. We
# arrange the headers to tell us the old tree was under directory
# `linux-old', and the new tree under `linux'.
# $_[0]: stegfs package root (like /tmp/sensors)
# $_[1]: Linux kernel tree (like /usr/src/linux)
# $_[2]: Name of the kernel file
# $_[3]: Name of the patched file
sub print_diff
{
  my ($package_root,$kernel_root,$kernel_file,$package_file) = @_;
  my ($diff_command,$dummy);

  $diff_command = "diff -u2";
  if ( -e "$kernel_root/$kernel_file") {
    $diff_command .= " $kernel_root/$kernel_file ";
  } else {
    $diff_command .= " /dev/null ";
  }
  if ( -e "$package_root/$package_file") {
    $diff_command .= " $package_root/$package_file ";
  } else {
    $diff_command .= " /dev/null";
  }
  open INPUT, "$diff_command|" or die "Can't execute `$diff_command'";
  $dummy = <INPUT>;
  $dummy = <INPUT>;
  print "--- linux-old/$kernel_file\t".`date`;
  print "+++ linux/$kernel_file\t".`date`;
    
  while (<INPUT>) {
    print;
  }
  close INPUT;
}

# This generates diffs for kernel file Documentation/Configure.help. This
# file contains the help texts that can be displayed during `make *config'
# for the kernel.
# The new texts are put at the end of the file, or just before the
# lm_sensors texts.
# Of course, care is taken old lines are removed.
# $_[0]: stegfs package root (like /tmp/i2c)
# $_[1]: Linux kernel tree (like /usr/src/linux)
sub gen_Documentation_Configure_help
{
  my ($package_root,$kernel_root) = @_;
  my $kernel_file = "Documentation/Configure.help";
  my $package_file = $temp;
  my $done = 0;

  open INPUT,"$kernel_root/$kernel_file"
        or die "Can't open `$kernel_root/$kernel_file'";
  open OUTPUT,">$package_root/$package_file"
        or die "Can't open $package_root/$package_file";
  while(<INPUT>) {

    if (! $done and m@^UFS@) {
      $done = 1;
      print OUTPUT <<'EOF'
StegFS steganographic filesystem support (experimental)
CONFIG_STEGFS_FS
  StegFS is an experimental filesystem based on the standard Linux
  second extended filesystem.

  This filesystem can be used in place of the ext2 filesystem. It has
  the addition that data can be securely hidden in parts of the disk
  which the main filesystem is not currently using. It scatters this
  hidden data over the space available. It encrypts data using a
  choice of algorithms.

  See the paper "The Steganographic Filesystem" by Anderson, Needham
  and Shamir for more background information. The paper "StegFS: A
  Steganographic File System for Linux" by McDonald and Kuhn describes
  this implementation.

  The homepage for this filesystem is:
    http://ban.joh.cam.ac.uk/~adm36/StegFS/

  If you want to compile this filesystem as a module ( = code which
  can be inserted in and removed from the running kernel whenever you
  want), say M here and read Documentation/modules.txt. The module
  will be called stegfs.o.

EOF
    }
    print OUTPUT;
  }
  close INPUT;
  close OUTPUT;
  die "Automatic patch generation for `Documentation/Configure.help' failed.\n".
      "Contact the authors please!" if $done == 0;
  print_diff $package_root,$kernel_root,$kernel_file,$package_file;
}

# This generates diffs for fs/Makefile
# First, `stegfs' is added to the ALL_SUB_DIRS list. Next, a couple of lines
# to add stegfs to the SUB_DIRS and/or MOD_SUB_DIRS lists is put right before
# Rules.make is included.
# $_[0]: stegfs package root (like /tmp/sensors)
# $_[1]: Linux kernel tree (like /usr/src/linux)
sub gen_fs_Makefile
{
  my ($package_root,$kernel_root) = @_;
  my $kernel_file = "fs/Makefile";
  my $package_file = $temp;
  my $pr1 = 0;
  my $pr2 = 0;
  my $new_style = 0;

  open INPUT,"$kernel_root/$kernel_file"
        or die "Can't open `$kernel_root/$kernel_file'";
  open OUTPUT,">$package_root/$package_file"
        or die "Can't open $package_root/$package_file";
  MAIN: while(<INPUT>) {
    if (m@^mod-subdirs\s*:=@) {
      $new_style = 1;
    }
    if (! $new_style and m@^ALL_SUB_DIRS\s*=@) {
      $pr1 = 1;
      while (m@\\$@) {
        print OUTPUT;
        $_ = <INPUT>;
      }
      s@$@ stegfs@;
      print OUTPUT;
      $_ = <INPUT>;
      redo MAIN;
    } 
    if (m@^include \$\(TOPDIR\)/Rules.make$@) {
      $pr2 = 1;
      if ($new_style) {
      print OUTPUT <<'EOF';
subdir-$(CONFIG_STEGFS_FS) 	+= stegfs
EOF
      } else {
      print OUTPUT <<'EOF';
ifeq ($(CONFIG_STEGFS_FS),y)
SUB_DIRS += stegfs
else
  ifeq ($(CONFIG_STEGFS_FS),m)
  MOD_SUB_DIRS += stegfs
  endif
endif

EOF
      }
    }
    print OUTPUT;
  }
  close INPUT;
  close OUTPUT;
  die "Automatic patch generation for `fs/Makefile' failed.\n".
      "Contact the authors please!" if $pr1 == 0 or $pr2 == 0;
  print_diff $package_root,$kernel_root,$kernel_file,$package_file;
}

# This generates diffs for fs/Config.in
# $_[0]: stegfs package root (like /tmp/sensors)
# $_[1]: Linux kernel tree (like /usr/src/linux)
sub gen_fs_Config_in
{
  my ($package_root,$kernel_root) = @_;
  my $kernel_file = "fs/Config.in";
  my $package_file = $temp;
  my $pr1 = 0;
#  my $kcrypto = 0;

  open INPUT,"$kernel_root/$kernel_file"
        or die "Can't open `$kernel_root/$kernel_file'";
  open OUTPUT,">$package_root/$package_file"
        or die "Can't open $package_root/$package_file";
#  if (-f "$kernel_root/crypto/Config.in") {
#    $kcrypto = 1;
#  }
  MAIN: while(<INPUT>) {
    if (m@CONFIG_NET@) {
      $pr1 = 1;
#      if ($kcrypto) {
#        print OUTPUT "dep_tristate 'StegFS steganographic filesystem (EXPERIMENTAL)' CONFIG_STEGFS_FS \$CONFIG_EXPERIMENTALi \$CONFIG_CIPHERS \$CONFIG_DIGEST\n\n";
#      }
#      else {
        print OUTPUT "dep_tristate 'StegFS steganographic filesystem (EXPERIMENTAL)' CONFIG_STEGFS_FS \$CONFIG_EXPERIMENTAL\n\n";
#      }
    }
    print OUTPUT;
  }
  close INPUT;
  close OUTPUT;
  die "Automatic patch generation for `fs/Config.in' failed.\n".
      "Contact the authors please!" if $pr1 == 0;
  print_diff $package_root,$kernel_root,$kernel_file,$package_file;
}
 

# This generates diffs for fs/filesystems.c They are a bit intricate.
# $_[0]: stegfs package root (like /tmp/sensors)
# $_[1]: Linux kernel tree (like /usr/src/linux)
sub gen_fs_filesystems_c
{
  my ($package_root,$kernel_root) = @_;
  my $kernel_file = "fs/filesystems.c";
  my $package_file = $temp;
  my $infunc = 0;
  my $done = 0;
  my $pr1 = 0;
  my $pr2 = 0;

  open INPUT,"$kernel_root/$kernel_file"
        or die "Can't open `$kernel_root/$kernel_file'";
  open OUTPUT,">$package_root/$package_file"
        or die "Can't open $package_root/$package_file";
  MAIN: while(<INPUT>) {
    if (m@devpts_fs\.h@) {
      $pr1 = 1;
      print OUTPUT << 'EOF';
#include <linux/stegfs_fs.h>
EOF
    }
    $infunc = 1 if (m@filesystem_setup@ and $pr1);
    if ($infunc and m@\}@) {
      $pr2 = 1;
      $infunc = 0;
      print OUTPUT <<'EOF';

#ifdef CONFIG_STEGFS_FS
	init_stegfs_fs();
#endif
EOF
      $done = 1;
    }
    print OUTPUT;
  }
  close INPUT;
  close OUTPUT;
  die "Automatic patch generation for `fs/filesystems.c' failed.\n".
      "Contact the authors please!" if $pr1 != $pr2;
  print_diff $package_root,$kernel_root,$kernel_file,$package_file;
}

# This generates diffs for include/linux/fs.h They are a bit intricate.
# $_[0]: stegfs package root (like /tmp/sensors)
# $_[1]: Linux kernel tree (like /usr/src/linux)
sub gen_include_linux_fs_h
{
  my ($package_root,$kernel_root) = @_;
  my $kernel_file = "include/linux/fs.h";
  my $package_file = $temp;
  my $pr1 = 0;
  my $pr2 = 0;
  my $pr3 = 0;
  my $pr4 = 0;

  open INPUT,"$kernel_root/$kernel_file"
        or die "Can't open `$kernel_root/$kernel_file'";
  open OUTPUT,">$package_root/$package_file"
        or die "Can't open $package_root/$package_file";
  MAIN: while(<INPUT>) {
    if (m@qnx4_fs_i\.h@) {
      $pr1 = 1;
      print OUTPUT << 'EOF';
#include <linux/stegfs_fs_i.h>
EOF
    }
    if (m@qnx4_inode_info@) {
      $pr2 = 1;
      print OUTPUT << 'EOF';
		struct stegfs_inode_info	stegfs_i;
EOF
    }
    if (m@qnx4_fs_sb\.h@) {
      $pr3 = 1;
      print OUTPUT << 'EOF';
#include <linux/stegfs_fs_sb.h>
EOF
    }
    if (m@qnx4_sb_info@) {
      $pr4 = 1;
      print OUTPUT << 'EOF';
		struct stegfs_sb_info	stegfs_sb;
EOF
    }
    print OUTPUT;
  }
  close INPUT;
  close OUTPUT;
  die "Automatic patch generation for `include/linux/fs.h' failed.\n".
      "Contact the authors please!" if $pr1 = 0 or $pr2 = 0 or $pr3 = 0 or $pr4 = 0;
  print_diff $package_root,$kernel_root,$kernel_file,$package_file;
}

sub gen_fs_stegfs_Makefile
{
  my ($package_root,$kernel_root) = @_;
  my $kernel_file = "fs/stegfs/Makefile";
  my $package_file = $temp;
  my $use_new_format;
  `grep -q -s 'obj-y' "$kernel_root/fs/Makefile"`;
  $use_new_format = ! $?;

  open OUTPUT,">$package_root/$package_file"
        or die "Can't open $package_root/$package_file";
  if ($use_new_format) {
      die "StegFS doesn't work on 2.4 kernels yet";
  } else {
    print OUTPUT <<'EOF';
#
# Makefile for the linux stegfs-filesystem routines.
#
# Note! Dependencies are done automagically by 'make dep', which also
# removes any old dependencies. DON'T put your own dependencies here
# unless it's something special (ie not a .c file).
#
# Note 2! The CFLAGS definitions are now in the main makefile...

O_TARGET := stegfs.o
O_OBJS   := acl.o balloc.o bitmap.o dir.o file.o fsync.o ialloc.o inode.o \
		ioctl.o mmap.o namei.o super.o symlink.o truncate.o \
		level.o
ifneq ($(CONFIG_CIPHERS),y)
  ifneq ($(CONFIG_CIPHERS),m)
    O_OBJS += aes.o serpent.o rc6.o twofish.o mars.o
  endif
endif
ifneq ($(CONFIG_DIGEST),y)
  ifneq ($(CONFIG_DIGEST),m)
    O_OBJS += sha1.o
  endif
endif
M_OBJS   := $(O_TARGET)

include $(TOPDIR)/Rules.make

EOF
  }  
  close OUTPUT;
  print_diff $package_root,$kernel_root,$kernel_file,$package_file;
}

# Generate the diffs for the list of MAINTAINERS
# $_[0]: stegfs package root (like /tmp/i2c)
# $_[1]: Linux kernel tree (like /usr/src/linux)
sub gen_MAINTAINERS
{
  my ($package_root,$kernel_root) = @_;
  my $kernel_file = "MAINTAINERS";
  my $package_file = $temp;
  my $done = 0;

  open INPUT,"$kernel_root/$kernel_file"
        or die "Can't open `$kernel_root/$kernel_file'";
  open OUTPUT,">$package_root/$package_file"
        or die "Can't open $package_root/$package_file";
  MAIN: while(<INPUT>) {
    if (not $done and (m@SVGA HANDLING@)) {
      print OUTPUT <<'EOF';
STEGFS FILESYSTEM
P:      Andrew McDonald
M:      andrew@mcdonald.org.uk
W:      http://ban.joh.cam.ac.uk/~adm36/StegFS/
S:      Maintained

EOF
      $done = 1;
    }
    print OUTPUT;
  }
  close INPUT;
  close OUTPUT;
  die "Automatic patch generation for `MAINTAINERS' failed.\n".
      "Contact the authors please!" if $done == 0;
  print_diff $package_root,$kernel_root,$kernel_file,$package_file;
}


# Main function
sub main
{
  my ($package_root,$kernel_root,%files,%includes,$package_file,$kernel_file);
  my ($diff_command,$dummy,$data0,$data1,$sedscript,$version_string);

  # --> Read the command-lineo
  $package_root = $ARGV[0];
  die "Package root `$package_root' is not found\n" 
        unless -d "$package_root/mkpatch";
  $kernel_root = $ARGV[1];
  die "Kernel root `$kernel_root' is not found\n" 
        unless -f "$kernel_root/Rules.make";

  # --> Read FILES
  open INPUT, "$package_root/mkpatch/FILES" 
        or die "Can't open `$package_root/mkpatch/FILES'";
  while (<INPUT>) {
    ($data0,$data1) = /(\S+)\s+(\S+)/;
    $files{$data0} = $data1;
  } 
  close INPUT;

  # --> Read INCLUDES
  open INPUT, "$package_root/mkpatch/INCLUDES" 
        or die "Can't open `$package_root/mkpatch/INCLUDES'";
  while (<INPUT>) {
    ($data0,$data1) = /(\S+)\s+(\S+)/;
    $includes{$data0} = $data1;
    $sedscript .= 's,(#\s*include\s*)'.$data0.'(\s*),\1'."$data1".'\2, ; ';
  } 
  close INPUT;

  # --> Read "version.h"
  open INPUT, "$package_root/version.h"
      or die "Can't open `$package_root/version.h'";
  $version_string .= $_ while <INPUT>;
  close INPUT;

  # --> Start generating
  foreach $package_file (sort keys %files) {
    open INPUT,"$package_root/$package_file" 
          or die "Can't open `$package_root/$package_file'";
    open OUTPUT,">$package_root/$temp"
          or die "Can't open `$package_root/$temp'";
    while (<INPUT>) {
      eval $sedscript;
      if (m@#\s*include\s*"../version.h"@) {
        print OUTPUT $version_string;
      } else {
        print OUTPUT;
      }
    }
    close INPUT;
    close OUTPUT;

    $kernel_file = $files{$package_file};
    print_diff $package_root,$kernel_root,$kernel_file,$temp;
  }

  gen_fs_Makefile $package_root, $kernel_root;
  gen_fs_Config_in $package_root, $kernel_root;
  gen_fs_filesystems_c $package_root, $kernel_root;
  gen_include_linux_fs_h $package_root, $kernel_root;
  gen_fs_stegfs_Makefile $package_root, $kernel_root;
  gen_Documentation_Configure_help $package_root, $kernel_root;
  gen_MAINTAINERS $package_root, $kernel_root;
}

main;

