/*
 *  linux/fs/stegfs/symlink.c
 *
 * Copyright (C) 1999
 * Andrew McDonald (andrew@mcdonald.org.uk)
 * St. John's College, University of Cambridge, UK
 *
 *  from
 *
 *  linux/fs/ext2/symlink.c
 *
 *  Copyright (C) 1992, 1993, 1994, 1995
 *  Remy Card (card@masi.ibp.fr)
 *  Laboratoire MASI - Institut Blaise Pascal
 *  Universite Pierre et Marie Curie (Paris VI)
 *
 *  from
 *
 *  linux/fs/minix/symlink.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 *
 *  ext2 symlink handling code
 */

#include "fs.h"
#include "stegfs_fs.h"

#include <asm/uaccess.h>

#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/stat.h>
#include <linux/malloc.h>

static int stegfs_readlink (struct dentry *, char *, int);
static struct dentry *stegfs_follow_link(struct dentry *, struct dentry *, unsigned int);

/*
 * symlinks can't do much...
 */
struct inode_operations stegfs_symlink_inode_operations = {
	NULL,			/* no file-operations */
	NULL,			/* create */
	NULL,			/* lookup */
	NULL,			/* link */
	NULL,			/* unlink */
	NULL,			/* symlink */
	NULL,			/* mkdir */
	NULL,			/* rmdir */
	NULL,			/* mknod */
	NULL,			/* rename */
	stegfs_readlink,		/* readlink */
	stegfs_follow_link,	/* follow_link */
	NULL,			/* readpage */
	NULL,			/* writepage */
	NULL,			/* bmap */
	NULL,			/* truncate */
	NULL,			/* permission */
	NULL			/* smap */
};

static struct dentry * stegfs_follow_link(struct dentry * dentry,
					struct dentry *base,
					unsigned int follow)
{
	struct inode *inode = dentry->d_inode;
	struct buffer_head * bh = NULL;
	int error;
	char * link;
	unsigned short iv;

	link = (char *) inode->u.stegfs_i.i_x->i_data;
	if (inode->i_blocks) {
		if (!(bh = stegfs_bread (inode, 0, &iv, 0, &error))) {
			dput(base);
			return ERR_PTR(-EIO);
		}

		if (STEGFS_IS_HID_INO(inode->i_ino)) {
			link = kmalloc (bh->b_size, GFP_KERNEL);
			stegfs_decrypt_cbc2(inode->i_sb,
					    STEGFS_INO_LVL(inode->i_ino),
					    bh->b_blocknr,
					    bh->b_data, link, bh->b_size, iv);
		}
		else
			link = bh->b_data;
	}
	UPDATE_ATIME(inode);
	base = lookup_dentry(link, base, follow);
	if (bh) {
		brelse(bh);
		if (STEGFS_IS_HID_INO(inode->i_ino))
			kfree (link);
	}
	return base;
}

static int stegfs_readlink (struct dentry * dentry, char * buffer, int buflen)
{
	struct inode *inode = dentry->d_inode;
	struct buffer_head * bh = NULL;
	char * link;
	int i;
	unsigned short iv;

	if (buflen > inode->i_sb->s_blocksize - 1)
		buflen = inode->i_sb->s_blocksize - 1;

	link = (char *) inode->u.stegfs_i.i_x->i_data;
	if (inode->i_blocks) {
		int err;
		bh = stegfs_bread (inode, 0, &iv, 0, &err);
		if (!bh) {
			if(err < 0) /* indicate type of error */
				return err;
			return 0;
		}
		if (STEGFS_IS_HID_INO(inode->i_ino)) {
			link = kmalloc (bh->b_size, GFP_KERNEL);
			stegfs_decrypt_cbc2(inode->i_sb,
					    STEGFS_INO_LVL(inode->i_ino),
					    bh->b_blocknr,
					    bh->b_data, link, bh->b_size, iv);
		}
		else
			link = bh->b_data;
	}

	i = 0;
	while (i < buflen && link[i])
		i++;
	if (copy_to_user(buffer, link, i))
		i = -EFAULT;
 	UPDATE_ATIME(inode);
	if (bh) {
		brelse (bh);
		if (STEGFS_IS_HID_INO(inode->i_ino))
			kfree (link);
	}
	return i;
}
