/*
 *  linux/fs/stegfs/mmap.c
 *
 * Copyright (C) 1999
 * Andrew McDonald (andrew@mcdonald.org.uk)
 * St. John's College, University of Cambridge, UK
 *
 *  from
 *
 *  linux/fs/smbfs/file.c
 *
 *  Copyright (C) 1995, 1996, 1997 by Paal-Kr. Engstad and Volker Lendecke
 *  Copyright (C) 1997 by Volker Lendecke
 *
 */

#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/fcntl.h>
#include <linux/stat.h>
#include <linux/mm.h>
#include <linux/malloc.h>
#include <linux/pagemap.h>

#include <asm/uaccess.h>
#include <asm/system.h>

#include "stegfs_fs.h"

#define STEGFS_EXTRAERR 1


static inline void stegfs_unlock_page(struct page *page)
{
	clear_bit(PG_locked, &page->flags);
	wake_up(&page->wait);
}


/*
 * Read a page synchronously.
 */
static int stegfs_readpage_sync(struct file *file, struct page *page)
{
	struct dentry *dentry = file->f_dentry;
	char *buffer = (char *) page_address(page);
	loff_t offset = page->offset;
	int result;
	mm_segment_t old_fs;

	clear_bit(PG_error, &page->flags);

	old_fs = get_fs();
	set_fs(KERNEL_DS);
	result = stegfs_file_read(file, buffer, PAGE_SIZE, &offset);
	set_fs(old_fs);
	if (result < 0) {
		goto io_error;
	}

	dentry->d_inode->i_atime = CURRENT_TIME;

	memset(buffer+result, 0, PAGE_SIZE-result);
	set_bit(PG_uptodate, &page->flags);
	result = 0;

io_error:
	stegfs_unlock_page(page);
	return result;
}

int stegfs_readpage(struct file *file, struct page *page)
{
	struct dentry *dentry = file->f_dentry;
	int error;

	stegfs_debug("inode: %lu\n", dentry->d_inode->i_ino);
	if (!STEGFS_IS_HID_INO(dentry->d_inode->i_ino))
		return generic_readpage(file, page);

#ifdef STEGFS_EXTRAERR
	if (test_bit(PG_locked, &page->flags))
		stegfs_debug("page already locked!\n");
#endif
	set_bit(PG_locked, &page->flags);
	set_bit(PG_free_after, &page->flags);
	atomic_inc(&page->count);
	error = stegfs_readpage_sync(file, page);
	free_page(page_address(page));
	return error;
}
