/*
 *  linux/fs/stegfs/ioctl.c
 *
 * Copyright (C) 1999-2001
 * Andrew McDonald (andrew@mcdonald.org.uk)
 *
 *  from
 *
 *  linux/fs/ext2/ioctl.c
 *
 *  Copyright (C) 1993, 1994, 1995
 *  Remy Card (card@masi.ibp.fr)
 *  Laboratoire MASI - Institut Blaise Pascal
 *  Universite Pierre et Marie Curie (Paris VI)
 */

#include "fs.h"
#include "stegfs_fs.h"

#include <asm/uaccess.h>

#include <linux/errno.h>
#include <linux/ioctl.h>
#include <linux/sched.h>
#include <linux/mm.h>

int stegfs_ioctl (struct inode * inode, struct file * filp, unsigned int cmd,
		unsigned long arg)
{
	unsigned int flags;
	char tmp;
	int i, result;
	/*struct stegfs_levelset levelset;*/
	struct stegfs_setlevel setlevel;

	stegfs_debug ("cmd = %u, arg = %lu\n", cmd, arg);

	switch (cmd) {
	case EXT2_IOC_GETFLAGS:
		flags = inode->u.stegfs_i.i_flags & EXT2_FL_USER_VISIBLE;
		return put_user(inode->u.stegfs_i.i_flags, (int *) arg);
	case EXT2_IOC_SETFLAGS:
		if (get_user(flags, (int *) arg))
			return -EFAULT;
		flags = flags & EXT2_FL_USER_MODIFIABLE;
		/*
		 * The IMMUTABLE and APPEND_ONLY flags can only be changed by
		 * a process with the relevant capability.
		 */
		if ((flags & (EXT2_APPEND_FL | EXT2_IMMUTABLE_FL)) ^
		    (inode->u.stegfs_i.i_flags &
		     (EXT2_APPEND_FL | EXT2_IMMUTABLE_FL)))
			/* This test looks nicer. Thanks to Pauline Middelink */
			if (!capable(CAP_LINUX_IMMUTABLE))
				return -EPERM;

		if ((current->fsuid != inode->i_uid) && !capable(CAP_FOWNER))
			return -EPERM;
		if (IS_RDONLY(inode))
			return -EROFS;
		inode->u.stegfs_i.i_flags = (inode->u.stegfs_i.i_flags &
					   ~EXT2_FL_USER_MODIFIABLE) | flags;
		if (flags & EXT2_SYNC_FL)
			inode->i_flags |= MS_SYNCHRONOUS;
		else
			inode->i_flags &= ~MS_SYNCHRONOUS;
		if (flags & EXT2_APPEND_FL)
			inode->i_flags |= S_APPEND;
		else
			inode->i_flags &= ~S_APPEND;
		if (flags & EXT2_IMMUTABLE_FL)
			inode->i_flags |= S_IMMUTABLE;
		else
			inode->i_flags &= ~S_IMMUTABLE;
		if (flags & EXT2_NOATIME_FL)
			inode->i_flags |= MS_NOATIME;
		else
			inode->i_flags &= ~MS_NOATIME;
		inode->i_ctime = CURRENT_TIME;
		mark_inode_dirty(inode);
		return 0;
	case EXT2_IOC_GETVERSION:
		return put_user(inode->i_generation, (int *) arg);
	case EXT2_IOC_SETVERSION:
		if ((current->fsuid != inode->i_uid) && !capable(CAP_FOWNER))
			return -EPERM;
		if (IS_RDONLY(inode))
			return -EROFS;
		if (get_user(inode->i_generation, (int *) arg))
			return -EFAULT;
		inode->i_ctime = CURRENT_TIME;
		mark_inode_dirty(inode);
		return 0;
	case STEGFS_IOC_GETICOPIES:
		if (STEGFS_IS_HID_INO(inode->i_ino))
			return put_user(inode->u.stegfs_i.i_icopies,
					(char *) arg);
		else
			return -ENOTTY;
	case STEGFS_IOC_SETICOPIES:
		if (!STEGFS_IS_HID_INO(inode->i_ino))
			return -ENOTTY;
		if ((current->fsuid != inode->i_uid) && !capable(CAP_FOWNER))
			return -EPERM;
		if (IS_RDONLY(inode))
			return -EROFS;
		if (get_user(tmp, (char *) arg))
			return -EFAULT;
		if (tmp < 1 || tmp > STEGFS_MAX_INO_COPIES)
			return -EINVAL;
		if (tmp < inode->u.stegfs_i.i_icopies) {
			for (i=tmp; i<inode->u.stegfs_i.i_icopies; i++) {
				stegfs_free_blocks(inode,
						   inode->u.stegfs_i.i_x->i_inode[i],
						   1);
				inode->u.stegfs_i.i_x->i_inode[i] = 0;
			}
		}
		inode->u.stegfs_i.i_icopies = tmp;
		inode->i_ctime = CURRENT_TIME;
		mark_inode_dirty(inode);
		return 0;
	case STEGFS_IOC_GETBCOPIES:
		if (STEGFS_IS_HID_INO(inode->i_ino))
			return put_user(inode->u.stegfs_i.i_bcopies,
					(char *) arg);
		else
			return -ENOTTY;
	case STEGFS_IOC_SETBCOPIES:
		if (!STEGFS_IS_HID_INO(inode->i_ino))
			return -ENOTTY;
		if ((current->fsuid != inode->i_uid) && !capable(CAP_FOWNER))
			return -EPERM;
		if (IS_RDONLY(inode))
			return -EROFS;
		if (get_user(tmp, (char *) arg))
			return -EFAULT;
		if (tmp < 1 || tmp > STEGFS_MAX_BLOCK_COPIES)
			return -EINVAL;
		if (tmp < inode->u.stegfs_i.i_bcopies) {
			/*
			for (i=tmp; i<inode->u.stegfs_i.i_icopies; i++)
				inode->u.stegfs_i.i_inode[i] = 0;
			*/
			/* FIXME: overwrite on delete */
		}
		inode->u.stegfs_i.i_bcopies = tmp;
		inode->i_ctime = CURRENT_TIME;
		mark_inode_dirty(inode);
		return 0;
	case STEGFS_IOC_GETLEVEL:
		printk("StegFS: Call to deprecated ioctl STEGFS_IOC_GETLEVEL\n");
		/* Support for this now removed */
#if 0
		if (inode->i_ino == EXT2_ROOT_INO &&
		    inode->i_sb->u.stegfs_sb.s_x->s_btabver < 2) {
			for (i=0; i<STEGFS_MAX_LEVELS-1; i++) {
				if (!inode->i_sb->u.stegfs_sb.s_x->s_slkeys[i])
					return put_user(i, (int *) arg);
			}
			return put_user(STEGFS_MAX_LEVELS-1, (int *) arg);
		}
#endif
		return -ENOTTY;
	case STEGFS_IOC_GETLEVELS:
		if (inode->i_ino == EXT2_ROOT_INO) {
			result = 0;
			for (i=0; i<(STEGFS_MAX_LEVELS-1); i++) {
				if(inode->i_sb->u.stegfs_sb.s_x->s_slkeys[i])
					result += (1 << i);
			}
			return put_user(result, (int *) arg);
		}
		else
			return -ENOTTY;
	case STEGFS_IOC_SETLEVEL:
		printk("StegFS: Call to deprecated ioctl STEGFS_IOC_SETLEVEL\n");
		/* Support for this now removed */
#if 0
		if (inode->i_ino != EXT2_ROOT_INO)
			return -ENOTTY;
/*
		if (inode->i_sb->u.stegfs_sb.s_btabver != 1)
			return -ENOTTY;
*/
		if (copy_from_user ((char *) &levelset, (char *) arg,
				sizeof (struct stegfs_levelset)))
			return -EFAULT;
		result = STEGFS_MAX_LEVELS - 1;
		for (i=0; i<(STEGFS_MAX_LEVELS-1); i++) {
			if(!inode->i_sb->u.stegfs_sb.s_x->s_slkeys[i]) {
				result = i;
				break;
			}
		}
		if (levelset.level < 0 || levelset.level > STEGFS_MAX_LEVELS-1)
			return -EINVAL;
		levelset.passw[255] = '\0'; /* make sure it is
							     zero terminated */
		stegfs_debug("level: %d password: *%s*\n", levelset.level,
			     levelset.passw);
                if (levelset.level < result) {
                        /* close level(s) */
                        memset(levelset.passw, 0, 256);
                        for (i=result; i>levelset.level; i--) {
                                stegfs_debug("close level %d\n", i);
                                result = stegfs_close_level(inode->i_sb, i);
                                if(!result) {
                                        stegfs_debug("level close failure\n");
                                        return -EPERM;
                                }
                        }
                        return 0;
                }
                else if(levelset.level == result) {
                        memset(levelset.passw, 0, 256);
                        return 0;
                }
                else if(levelset.level > result) {
                        for (i=result+1; i<levelset.level+1; i++) {
                                stegfs_debug("open level %d\n", i);
                                result = stegfs_open_level(inode->i_sb, i,
                                                           levelset.level,
                                                           levelset.passw);
                                stegfs_debug("solresult: %d\n", result);
                                if(!result ||
                                   !STEGFS_SB(inode->i_sb)->s_x->s_slkeys[i-1]) {
                                        stegfs_debug("level open failure\n");
                                        memset(levelset.passw, 0, 256);
                                        return -EBADPASS;
                                }
                        }
                        memset(levelset.passw, 0, 256);
                        return 0;
                }
#endif
		return -ENOTTY;
	case STEGFS_IOC_SETLEVELS:
		if (inode->i_ino != EXT2_ROOT_INO)
			return -ENOTTY;
		if ((current->fsuid != inode->i_uid) && !capable(CAP_FOWNER))
			return -EPERM;
		if (copy_from_user ((char *) &setlevel, (char *) arg,
				sizeof (struct stegfs_setlevel)))
			return -EFAULT;
		if (setlevel.level < 0 || setlevel.level > STEGFS_MAX_LEVELS-1)
			return -EINVAL;
		if (setlevel.context < 0 || setlevel.context > STEGFS_MAX_CONTEXTS)
			return -EINVAL;

		setlevel.passw[STEGFS_PASS_LEN-1] = '\0'; /* make sure it is
							     zero terminated */
		stegfs_debug("context: %d level: %d flags: %d password: *%s*\n",
			     setlevel.context, setlevel.level,
			     setlevel.flags, setlevel.passw);
		if (setlevel.flags & 0x1) {
			/* close level */
			memset(setlevel.passw, 0, STEGFS_PASS_LEN);

			stegfs_debug("close level %d\n", setlevel.level);

			if(setlevel.level) {
				result = stegfs_close_level(inode->i_sb, setlevel.level);
				if(result != 1) {
					stegfs_debug("level close failure\n");
					return result;
				}
			}
			else {
				for (i=0; i<STEGFS_MAX_LEVELS-1; i++) {
					if (STEGFS_SB(inode->i_sb)->s_x->s_slkeys[i]) {
						result = stegfs_close_level(inode->i_sb, i+1);
						if(result != 1) {
							stegfs_debug("level close failure\n");
							return result;
						}
					}
				}
			}
			return 0;
		}
		else {
			if (setlevel.context < 1 ||
			    setlevel.context > STEGFS_MAX_CONTEXTS)
				return -EINVAL;
			if(setlevel.level) {
				stegfs_debug("open level %d\n", setlevel.level);
				result = stegfs_open_level(inode->i_sb, setlevel.level,
							   setlevel.context, setlevel.passw);
				stegfs_debug("solresult: %d\n", result);
				memset(setlevel.passw, 0, STEGFS_PASS_LEN);
				return 0;
			}
			else {
				for (i=1; i<STEGFS_MAX_LEVELS; i++) {
					if (!STEGFS_SB(inode->i_sb)->s_x->s_slkeys[i-1]) {
						stegfs_debug("open level %d\n", i);
						result = stegfs_open_level(inode->i_sb, i,
									   setlevel.context,
									   setlevel.passw);
						stegfs_debug("solresult: %d\n", result);
/*
						if(!result ||
						   !STEGFS_SB(inode->i_sb)->s_slkeys[i-1]) {
						   stegfs_debug("level open failure\n");
						   memset(setlevel.passw, 0, STEGFS_PASS_LEN);
						   return -EBADPASS;
						}
*/
					}
				}
				memset(setlevel.passw, 0, STEGFS_PASS_LEN);
				return 0;
			}
		}
		return -ENOTTY;
	default:
		return -ENOTTY;
	}
}
