/*
 *  linux/fs/stegfs/inode.c
 *
 * Copyright (C) 1999-2001
 * Andrew McDonald (andrew@mcdonald.org.uk)
 *
 *  from
 *
 *  linux/fs/ext2/inode.c
 *
 *  Copyright (C) 1992, 1993, 1994, 1995
 *  Remy Card (card@masi.ibp.fr)
 *  Laboratoire MASI - Institut Blaise Pascal
 *  Universite Pierre et Marie Curie (Paris VI)
 *
 *  from
 *
 *  linux/fs/minix/inode.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 *
 *  Goal-directed block allocation by Stephen Tweedie
 * 	(sct@dcs.ed.ac.uk), 1993, 1998
 *  Big-endian to little-endian byte-swapping/bitmaps by
 *        David S. Miller (davem@caip.rutgers.edu), 1995
 *  64-bit file support on 64-bit platforms by Jakub Jelinek
 * 	(jj@sunsite.ms.mff.cuni.cz)
 */

#include "fs.h"
#include "stegfs_fs.h"

#include <asm/uaccess.h>
#include <asm/system.h>

#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/stat.h>
#include <linux/string.h>
#include <linux/locks.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/random.h>

#if defined(CONFIG_DIGEST) || defined(CONFIG_DIGEST_MODULE)
#include <linux/crypto.h>
#else
#include "sha1.h"
#endif

static int stegfs_update_inode(struct inode * inode, int do_sync);

/*
 * Called at each iput()
 */
void stegfs_put_inode (struct inode * inode)
{
	stegfs_discard_prealloc (inode);
}

/*
 * Called when the VFS inode is no longer useful.
 */
void stegfs_clear_inode (struct inode * inode)
{
	stegfs_debug("clearing inode: %lu\n", inode->i_ino);

	if (!inode->u.stegfs_i.i_x) {
		/* this isn't necessarily an error
		   as long as we don't try to use the inode
		   in between calls!
		*//*
		stegfs_error(inode->i_sb, "stegfs_clear_inode",
			     "double clear of i_x");
		*/
		return;
	}
	kfree(inode->u.stegfs_i.i_x);
	inode->u.stegfs_i.i_x = NULL;
}

/*
 * Called at the last iput() if i_nlink is zero.
 */
void stegfs_delete_inode (struct inode * inode)
{
	if (is_bad_inode(inode) ||
	    inode->i_ino == EXT2_ACL_IDX_INO ||
	    inode->i_ino == EXT2_ACL_DATA_INO)
		return;
	inode->u.stegfs_i.i_dtime = CURRENT_TIME;
	/* When we delete an inode, we increment its i_generation.
	   If it is read in from disk again, the generation will differ. */
	inode->i_generation++;
	mark_inode_dirty(inode);
	stegfs_update_inode(inode, IS_SYNC(inode));
	inode->i_size = 0;
	if (inode->i_blocks)
		stegfs_truncate (inode);
	stegfs_free_inode (inode);
}

#define inode_bmap(inode, nr) le32_to_cpu((inode)->u.stegfs_i.i_x->i_data[(nr)])

/* 
 * stegfs_discard_prealloc and stegfs_alloc_block are atomic wrt. the
 * superblock in the same manner as are stegfs_free_blocks and
 * stegfs_new_block.  We just wait on the super rather than locking it
 * here, since stegfs_new_block will do the necessary locking and we
 * can't block until then.
 */
void stegfs_discard_prealloc (struct inode * inode)
{
#ifdef EXT2_PREALLOCATE
	unsigned short total;

	if (inode->u.stegfs_i.i_prealloc_count) {
		total = inode->u.stegfs_i.i_prealloc_count;
		inode->u.stegfs_i.i_prealloc_count = 0;
		stegfs_free_blocks (inode, inode->u.stegfs_i.i_prealloc_block, total);
	}
#endif
}


static int stegfs_alloc_block (struct inode * inode,
			       unsigned long goal, unsigned long ncopy,
			       int * err)
{
#ifdef STEGFS_DEBUG
	static unsigned long alloc_hits = 0, alloc_attempts = 0;
#endif
	unsigned long result;
	struct buffer_head * bh;

	wait_on_super (inode->i_sb);

#ifdef EXT2_PREALLOCATE
	if (inode->u.stegfs_i.i_prealloc_count &&
	    (goal == inode->u.stegfs_i.i_prealloc_block ||
	     goal + 1 == inode->u.stegfs_i.i_prealloc_block))
	{		
		result = inode->u.stegfs_i.i_prealloc_block++;
		inode->u.stegfs_i.i_prealloc_count--;
		stegfs_debug ("preallocation hit (%lu/%lu).\n",
			    ++alloc_hits, ++alloc_attempts);

		/* It doesn't matter if we block in getblk() since
		   we have already atomically allocated the block, and
		   are only clearing it now. */
		if (!(bh = getblk (inode->i_sb->s_dev, result,
				   inode->i_sb->s_blocksize))) {
			stegfs_error (inode->i_sb, "stegfs_alloc_block",
				    "cannot get block %lu", result);
			return 0;
		}
		if (!buffer_uptodate(bh))
			wait_on_buffer(bh);
		memset(bh->b_data, 0, inode->i_sb->s_blocksize);
		mark_buffer_uptodate(bh, 1);
		mark_buffer_dirty(bh, 1);
		brelse (bh);
	} else {
		stegfs_discard_prealloc (inode);
#ifdef STEGFS_DEBUG
		if (!STEGFS_IS_HID_INO(inode->i_ino))
			stegfs_debug ("preallocation miss (%lu/%lu).\n",
			    alloc_hits, ++alloc_attempts);
#endif
		if (S_ISREG(inode->i_mode))
			result = stegfs_new_block (inode, goal, 
				 &inode->u.stegfs_i.i_prealloc_count,
				 &inode->u.stegfs_i.i_prealloc_block, err);
		else
			result = stegfs_new_block (inode, goal, 0, 0, err);
	}
#else
	result = stegfs_new_block (inode, goal, 0, 0, err);
#endif

	return result;
}

/**
 *     stegfs_block_to_path - parse the block number into array of offsets
 *     @inode: inode in question (we are only interested in its superblock)
 *     @i_block: block number to be parsed
 *     @offsets: array to store the offsets in
 *
 *     To store the locations of file's data ext2 uses a data structure common
 *     for UNIX filesystems - tree of pointers anchored in the inode, with
 *     data blocks at leaves and indirect blocks in intermediate nodes.
 *     This function translates the block number into path in that tree -
 *     return value is the path length and @offsets[n] is the offset of
 *     pointer to (n+1)th node in the nth one. If @block is out of range
 *     (negative or too large) warning is printed and zero returned.
 *
 *     Note: function doesn't find node addresses, so no IO is needed. All
 *     we need to know is the capacity of indirect blocks (taken from the
 *     inode->i_sb).
 */

/*
 * Portability note: the last comparison (check that we fit into triple
 * indirect block) is spelled differently, because otherwise on an
 * architecture with 32-bit longs and 8Kb pages we might get into trouble
 * if our filesystem had 8Kb blocks. We might use long long, but that would
 * kill us on x86. Oh, well, at least the sign propagation does not matter -
 * i_block would have to be negative in the very beginning, so we would not
 * get there at all.
 */

static int stegfs_block_to_path(struct inode *inode, long i_block, int offsets[4])
{
	int ptrs = EXT2_ADDR_PER_BLOCK(inode->i_sb);
	int ptrs_bits = STEGFS_ADDR_PER_BLOCK_BITS(inode->i_sb);
	const long direct_blocks = EXT2_NDIR_BLOCKS,
		indirect_blocks = ptrs,
		double_blocks = (1 << (ptrs_bits * 2));
	int n = 0;

	if (i_block < 0) {
		stegfs_warning (inode->i_sb, "stegfs_block_to_path", "block < 0");
	} else if (i_block < direct_blocks) {
		offsets[n++] = i_block;
	} else if ( (i_block -= direct_blocks) < indirect_blocks) {
		offsets[n++] = EXT2_IND_BLOCK;
		offsets[n++] = i_block;
	} else if ((i_block -= indirect_blocks) < double_blocks) {
		offsets[n++] = EXT2_DIND_BLOCK;
		offsets[n++] = i_block >> ptrs_bits;
		offsets[n++] = i_block & (ptrs - 1);
	} else if (((i_block -= double_blocks) >> (ptrs_bits * 2)) < ptrs) {
		offsets[n++] = EXT2_TIND_BLOCK;
		offsets[n++] = i_block >> (ptrs_bits * 2);
		offsets[n++] = (i_block >> ptrs_bits) & (ptrs - 1);
		offsets[n++] = i_block & (ptrs - 1);
	} else {
		stegfs_warning (inode->i_sb, "stegfs_block_to_path", "block > big");
	}
	return n;
}

int stegfs_bmap (struct inode * inode, int block)
{
	int offsets[4], *p;
	int depth = stegfs_block_to_path(inode, block, offsets);
	struct buffer_head *bh;
	int i;

	if (STEGFS_IS_HID_INO(inode->i_ino)) {
		stegfs_warning (inode->i_sb, "stegfs_bmap",
				"cannot bmap for stegfs inode");
	}
	if (depth == 0)
		goto fail;
	i = inode_bmap (inode, *(p=offsets));
	while (--depth) {
		if (!i)
			break;
		bh = bread (inode->i_dev, i, inode->i_sb->s_blocksize);
		if (!bh)
			goto fail;
		++p;
		i = le32_to_cpu(((u32 *) bh->b_data)[*p]);
		brelse (bh);
	}
	return i;
fail:
	return 0;
}

static struct buffer_head * inode_getblkn (struct inode * inode, int nr,
					   unsigned short *iv,
					   int create, int new_block,
					   int * err, int ncopy, int *ncopy2, int recreate,
					   int * cerr)
{
	u32 * p;
	int tmp, goal = 0;
	struct buffer_head * result;
	int blocks = inode->i_sb->s_blocksize / 512;
	int i,c;
	struct stegfs_btable btab;
	unsigned short randpert = 0;

	if (!STEGFS_IS_HID_INO(inode->i_ino)) {
		p = inode->u.stegfs_i.i_x->i_data + nr;
		*iv = 0;
repeat:
		tmp = *p;
		if (tmp) {
			struct buffer_head * result;
			result = getblk (inode->i_dev, le32_to_cpu(tmp),
					 inode->i_sb->s_blocksize);
			if (tmp == *p)
				return result;
			brelse (result);
			goto repeat;
		}
		*err = -EFBIG;
		if (!create)
			return NULL;

		if (inode->u.stegfs_i.i_next_alloc_block == new_block)
			goal = inode->u.stegfs_i.i_next_alloc_goal;

		stegfs_debug ("hint = %d,", goal);
		
		if (!goal) {
			for (tmp = nr - 1; tmp >= 0; tmp--) {
				if (inode->u.stegfs_i.i_x->i_data[tmp]) {
					goal = le32_to_cpu(inode->u.stegfs_i.i_x->i_data[tmp]);
					break;
				}
			}
			if (!goal) {
				goal = (inode->u.stegfs_i.i_block_group * 
					STEGFS_BLOCKS_PER_GROUP(inode->i_sb)) +
					le32_to_cpu(inode->i_sb->
						    u.stegfs_sb.s_es->
						    s_first_data_block);

				get_random_bytes(&randpert,
						 sizeof(unsigned short));
				if (randpert & 0x7) {
					stegfs_debug("goal randpert: %u\n", (randpert >> 3));
					goal += (randpert >> 3);
				}

			}
		}

		stegfs_debug ("goal = %d.\n", goal);

		tmp = stegfs_alloc_block (inode, goal, 0, err);
		if (!tmp)
			return NULL;
		result = getblk (inode->i_dev, tmp, inode->i_sb->s_blocksize);
		if (*p) {
			stegfs_free_blocks (inode, tmp, 1);
			brelse (result);
			goto repeat;
		}
		*p = cpu_to_le32(tmp);
		inode->u.stegfs_i.i_next_alloc_block = new_block;
		inode->u.stegfs_i.i_next_alloc_goal = tmp;
		inode->i_ctime = CURRENT_TIME;
		inode->i_blocks += blocks;
		if (IS_SYNC(inode) || inode->u.stegfs_i.i_osync)
			stegfs_sync_inode (inode);
		else
			mark_inode_dirty(inode);
		return result;
	}
	/* else STEGFS_IS_HID_INO(inode) */
	p = inode->u.stegfs_i.i_x->i_data + nr;
	*ncopy2 = ncopy;
	if (ncopy == STEGFS_GB_ANYCOPY) {
		/* just want a valid copy */
		for(c = 0; c < inode->u.stegfs_i.i_bcopies; c++) {
			*ncopy2 = (c + inode->u.stegfs_i.i_pref_bcopy) % 
				inode->u.stegfs_i.i_bcopies;
			p = inode->u.stegfs_i.i_x->i_data + nr +
				*ncopy2 * EXT2_N_BLOCKS;
repeat2:
			tmp = *p;
/*
			stegfs_debug("Getting block: %u (%u)\n", nr, tmp);
*/
			if(tmp) {
				/* file is using this block */
				struct buffer_head * result =
					getblk (inode->i_dev, le32_to_cpu(tmp),
						inode->i_sb->s_blocksize);
				if(!buffer_uptodate(result)) {
					ll_rw_block (READ, 1, &result);
					wait_on_buffer (result);
					if (!buffer_uptodate(result)) {
						brelse(result);
						/* Hmm, what to do. */
						continue;
					}
				}
				if (tmp == *p) {
/*
                                        stegfs_debug("Got the block\n");
*/
					stegfs_get_btab(inode->i_sb, le32_to_cpu(*p), &btab);
					stegfs_decrypt_btab(inode->i_sb,
							    STEGFS_INO_LVL(inode->i_ino),
							    le32_to_cpu(*p), &btab);

					if(btab.magic1 == 0 &&
					   (btab.magic2 == 0 ||
					    btab.magic2 == 1) &&
/*						   btab.magic2 == *p && */
					   stegfs_chksum(result->b_data,
							 result->b_size)
					   == btab.bchecksum) {
/*
					        stegfs_debug("and the xsum is ok\n");
*/
						if(cerr)
							*cerr = STEGFS_GB_OK;
						inode->u.stegfs_i.i_pref_bcopy = *ncopy2;
						*iv = btab.iv;
						return result;
					}
					else {
/*
                                                stegfs_debug("bad xsum\n");

 	  stegfs_debug("%08x %08x %lu\n",
           btab.bchecksum, stegfs_chksum(result->b_data,
           result->b_size), result->b_size);
*/
						brelse (result);
						continue;
					}
				}
				stegfs_debug("how did we get here?\n");
				brelse (result);
				goto repeat2;
			}
			else {
				break;
			}
		}

		if(c == inode->u.stegfs_i.i_bcopies) {
			stegfs_warning(inode->i_sb, "stegfs_getblk",
				       "No valid copies of block found"
				       " (inode %lu)", inode->i_ino);
			*err = -EFBIG;
			return NULL;
		}
	}
	else {
		if (ncopy > inode->u.stegfs_i.i_bcopies) {
			if(cerr)
				*cerr = STEGFS_GB_NOCOPYN;
			*err = -EFBIG;
			return NULL;
		}

		p = inode->u.stegfs_i.i_x->i_data + nr + ncopy * EXT2_N_BLOCKS;

repeat3:
		tmp = *p;
		if(tmp) {
				/* file is using this block */
			struct buffer_head * result =
				getblk (inode->i_dev, le32_to_cpu(tmp),
					inode->i_sb->s_blocksize);

			stegfs_debug("getting block %u (copy %u)\n", *p, ncopy);

			if (tmp == *p) {
				stegfs_get_btab(inode->i_sb, le32_to_cpu(*p), &btab);
				stegfs_decrypt_btab(inode->i_sb,
						    STEGFS_INO_LVL(inode->i_ino),
						    le32_to_cpu(*p), &btab);
				if (cerr)
					*cerr = STEGFS_GB_OK;
				if (!buffer_uptodate(result)) {
					ll_rw_block (READ, 1, &result);
					wait_on_buffer (result);
					if (!buffer_uptodate(result)) {
						brelse (result);
						*err = -EIO;
						return NULL;
					}
				}
				if (btab.magic1 == 0 &&
				    (btab.magic2 == 0 || btab.magic2 == 1) &&
				    /* btab.magic2 == *p && */
				    stegfs_chksum(result->b_data,
						  result->b_size)
				    == btab.bchecksum) {
/*
						stegfs_debug("good xsum\n");
*/
					*iv = btab.iv;
					return result;
				}
				else {
					if(!recreate) {
/*
						stegfs_debug("bad xsum - not recreating\n");
*/
						if (cerr)
							*cerr = STEGFS_GB_NOVALID;
						brelse(result);
						*err = -EIO;
						return NULL;
					}

					stegfs_debug("bad xsum - recreating - magic1: %u\n", btab.magic1);
/*
					stegfs_debug("reallocating copy %u/%u\n", ncopy,
						     inode->u.stegfs_i.i_bcopies);
*/
					brelse(result);
					tmp = stegfs_alloc_block (inode,
								  inode->u.stegfs_i.i_x->i_next_goal_block[ncopy],
								  ncopy, err);
					if (!tmp)
						return NULL;
					result = getblk (inode->i_dev, tmp,
							 inode->i_sb->s_blocksize);
					if (*p) {
						stegfs_free_blocks (inode, le32_to_cpu(*p), 1);
					}
					*p = cpu_to_le32(tmp);
					inode->u.stegfs_i.i_x->i_next_goal_block[ncopy] = tmp+1;
					inode->i_ctime = CURRENT_TIME;
					if (cerr)
						*cerr = STEGFS_GB_RECREATED;
					if (IS_SYNC(inode) || inode->u.stegfs_i.i_osync)
						stegfs_sync_inode (inode);
					else
						mark_inode_dirty(inode);
					stegfs_get_btab(inode->i_sb, tmp, &btab);
					stegfs_decrypt_btab(inode->i_sb,
							    STEGFS_INO_LVL(inode->i_ino),
							    tmp, &btab);
					*iv = btab.iv;
					return result;
				}
			}
			brelse (result);
			goto repeat3;
		}
	}
	*err = -EFBIG;

	if (!create) {
		if(cerr)
			*cerr = STEGFS_GB_NOCOPYN;
		*err = -EFBIG;
		return NULL;
	}

	for(i=0; i<inode->u.stegfs_i.i_bcopies; i++) {
                
		stegfs_debug("allocating copy %u/%u\n", i+1,
			     inode->u.stegfs_i.i_bcopies);
		
		tmp = stegfs_alloc_block (inode,
					  inode->u.stegfs_i.i_x->i_next_goal_block[i],
					  i, err);
		if (!tmp)
			return NULL;
		result = getblk (inode->i_dev, tmp, inode->i_sb->s_blocksize);
		if (*(p+i*EXT2_N_BLOCKS)) {
			stegfs_free_blocks (inode, tmp, 1);
			brelse (result);
			continue;
		}
		*(p+i*EXT2_N_BLOCKS) = cpu_to_le32(tmp);
		stegfs_debug("used blk: %u\n", tmp);
		inode->u.stegfs_i.i_x->i_next_goal_block[i] = tmp+1;
		brelse (result);
	}

	result = getblk (inode->i_dev, le32_to_cpu(*p), inode->i_sb->s_blocksize);

	inode->i_ctime = CURRENT_TIME;
	inode->i_blocks += blocks;
	*ncopy2 = 0;
	if (cerr)
		*cerr = STEGFS_GB_CREATED;

	if (IS_SYNC(inode) || inode->u.stegfs_i.i_osync)
		stegfs_sync_inode (inode);
	else
		mark_inode_dirty(inode);
	stegfs_get_btab(inode->i_sb, le32_to_cpu(*p), &btab);
	stegfs_decrypt_btab(inode->i_sb,
			    STEGFS_INO_LVL(inode->i_ino),
			    le32_to_cpu(*p), &btab);
	*iv = btab.iv;
	return result;
}

#if 0
static struct buffer_head * inode_getblk (struct inode * inode, int nr,
					  int create, int new_block, int * err)
{
	return inode_getblkn(inode, nr, create, new_block, err, 0, 0, NULL);
}
#endif

static struct buffer_head * block_getblkn (struct inode * inode,
					   struct buffer_head * bh, int nr,
					   unsigned short *iv,
					   int create, int blocksize, 
					   int new_block, int * err,
					   int ncopy, int recreate, int * cerr)
{
	int tmp, goal = 0;
	u32 * p;
	struct buffer_head * result;
	int blocks = inode->i_sb->s_blocksize / 512;
	struct stegfs_btable btab;
	char * bufplain;

	if (!bh)
		return NULL;
	if (!buffer_uptodate(bh)) {
		ll_rw_block (READ, 1, &bh);
		wait_on_buffer (bh);
		if (!buffer_uptodate(bh)) {
			brelse (bh);
			return NULL;
		}
	}
	if (!STEGFS_IS_HID_INO(inode->i_ino)) {
		p = (u32 *) bh->b_data + nr;
		*iv = 0;
repeat:
		tmp = le32_to_cpu(*p);
		if (tmp) {
			result = getblk (bh->b_dev, tmp, blocksize);
			if (tmp == le32_to_cpu(*p)) {
				brelse (bh);
				return result;
			}
			brelse (result);
			goto repeat;
		}
		*err = -EFBIG;
		if (!create) {
			brelse (bh);
			return NULL;
		}

		if (inode->u.stegfs_i.i_next_alloc_block == new_block)
			goal = inode->u.stegfs_i.i_next_alloc_goal;
		if (!goal) {
			for (tmp = nr - 1; tmp >= 0; tmp--) {
				if (le32_to_cpu(((u32 *) bh->b_data)[tmp])) {
					goal = le32_to_cpu(((u32 *)bh->b_data)[tmp]);
					break;
				}
			}
			if (!goal)
				goal = bh->b_blocknr;
		}
		tmp = stegfs_alloc_block (inode, goal, 0, err);
		if (!tmp) {
			brelse (bh);
			return NULL;
		}
		result = getblk (bh->b_dev, tmp, blocksize);
		if (le32_to_cpu(*p)) {
			stegfs_free_blocks (inode, tmp, 1);
			brelse (result);
			goto repeat;
		}
		*p = le32_to_cpu(tmp);
		mark_buffer_dirty(bh, 1);
		if (IS_SYNC(inode) || inode->u.stegfs_i.i_osync) {
			ll_rw_block (WRITE, 1, &bh);
			wait_on_buffer (bh);
		}
		inode->i_ctime = CURRENT_TIME;
		inode->i_blocks += blocks;
		mark_inode_dirty(inode);
		inode->u.stegfs_i.i_next_alloc_block = new_block;
		inode->u.stegfs_i.i_next_alloc_goal = tmp;
		brelse (bh);
		return result;
	}
	/* else STEGFS_IS_HID_INO(inode)*/
	stegfs_debug("copy: %d\n", ncopy);
	bufplain = kmalloc(bh->b_size, GFP_KERNEL);
	if(!bufplain) {
		stegfs_debug("kmalloc failed\n");
		brelse(bh);
		return NULL;
	}
	stegfs_decrypt_cbc2(inode->i_sb,
			    STEGFS_INO_LVL(inode->i_ino),
			    bh->b_blocknr,
			    bh->b_data, bufplain, bh->b_size, *iv);
	p = (u32 *) bufplain + nr;
repeat2:
	tmp = le32_to_cpu(*p);
/*
	stegfs_debug("Getting block: %u\n", tmp);
*/
	if (tmp) {
		result = getblk (bh->b_dev, tmp, blocksize);
		if (tmp == le32_to_cpu(*p)) {
			if (!buffer_uptodate(result)) {
				ll_rw_block (READ, 1, &result);
				wait_on_buffer (result);
				if (!buffer_uptodate(result)) {
					brelse(bh);
					kfree(bufplain);
					brelse (result);
					return NULL;
				}
			}

			stegfs_get_btab(inode->i_sb, tmp, &btab);
			stegfs_decrypt_btab(inode->i_sb,
					    STEGFS_INO_LVL(inode->i_ino),
					    tmp, &btab);
			if (btab.magic1 == 0 &&
			    (btab.magic2 == 0 || btab.magic2 == 1) &&
			    stegfs_chksum(result->b_data,
					  result->b_size) == btab.bchecksum) {

				stegfs_debug("indb xsum ok\n");

				brelse (bh);
				kfree(bufplain);
				*iv = btab.iv;
				if (cerr)
					*cerr = STEGFS_GB_OK;
				return result;
			}
			else {


				stegfs_debug("indb bad xsum\n");

				if (!recreate) {
					brelse (bh);
					kfree(bufplain);
					if (cerr)
						*cerr = STEGFS_GB_NOVALID;
					*err = -EIO;
					brelse (result);
					return NULL;
				}
				else {
                                        stegfs_debug("bad xsum - recreating - magic1: %u\n", btab.magic1);
					brelse(result);

                                        tmp = stegfs_alloc_block (inode,
								  inode->u.stegfs_i.i_x->i_next_goal_block[ncopy],
								  ncopy, err);
                                        if (!tmp)
                                                return NULL;
                                        result = getblk (inode->i_dev, tmp,
                                                         inode->i_sb->s_blocksize);
                                        if (*p) {
                                                stegfs_free_blocks (inode, *p, 1);
                                        }
                                        *p = le32_to_cpu(tmp);
                                        inode->u.stegfs_i.i_x->i_next_goal_block[ncopy] = tmp+1;
                                        inode->i_ctime = CURRENT_TIME;

                                        if (cerr)
                                                *cerr = STEGFS_GB_RECREATED;
                                        if (IS_SYNC(inode) || inode->u.stegfs_i.i_osync)
                                                stegfs_sync_inode (inode);
                                        else
                                                mark_inode_dirty(inode);

					stegfs_get_btab(inode->i_sb, tmp, &btab);
					stegfs_decrypt_btab(inode->i_sb,
							    STEGFS_INO_LVL(inode->i_ino),
							    tmp, &btab);
					*iv = btab.iv;

                                        /* fix indirection block */        
					stegfs_encrypt_cbc2(inode->i_sb,
							    STEGFS_INO_LVL(inode->i_ino),
							    bh->b_blocknr,
							    bufplain, bh->b_data, bh->b_size, &btab.iv);
					mark_buffer_dirty(bh, 1);

					btab.magic1 = 0;
					btab.magic2 = 0;
					btab.ino = inode->i_ino;
					btab.bchecksum = stegfs_chksum(bh->b_data, bh->b_size);
					stegfs_encrypt_btab(inode->i_sb,
							    STEGFS_INO_LVL(inode->i_ino),
							    bh->b_blocknr, &btab);
					stegfs_put_btab(inode->i_sb, bh->b_blocknr, &btab);

					brelse (bh);
					kfree(bufplain);
                                        return result;


				}
			}
		}
		brelse (result);
		goto repeat2;
	}

	*cerr = STEGFS_GB_NOVALID;
	*err = -EFBIG;
	if (!create) {
		brelse (bh);
		kfree(bufplain);
		return NULL;
	}

	tmp = stegfs_alloc_block(inode,
				 inode->u.stegfs_i.i_x->i_next_goal_block[ncopy],
				 ncopy, err);
	if (!tmp) {
		brelse (bh);
		kfree(bufplain);
		return NULL;
	}
	result = getblk (bh->b_dev, tmp, blocksize);
	if (le32_to_cpu(*p)) {
		stegfs_free_blocks (inode, tmp, 1);
		brelse (result);
		goto repeat2;
	}
	*p = le32_to_cpu(tmp);

	/* Hmm, is this next bit right? */
	if (!buffer_uptodate(result)) {
		ll_rw_block (READ, 1, &result);
		wait_on_buffer (result);
		if (!buffer_uptodate(result)) {
			brelse (result);
			brelse (bh);
			kfree(bufplain);
			return NULL;
		}
	}
	stegfs_get_btab(inode->i_sb, tmp, &btab);
	stegfs_decrypt_btab(inode->i_sb,
			    STEGFS_INO_LVL(inode->i_ino),
			    tmp, &btab);
	*iv = btab.iv;
	inode->u.stegfs_i.i_x->i_next_goal_block[ncopy] = tmp+1;

	stegfs_encrypt_cbc2(inode->i_sb,
			    STEGFS_INO_LVL(inode->i_ino),
			    bh->b_blocknr,
			    bufplain, bh->b_data, bh->b_size, &btab.iv);
	mark_buffer_dirty(bh, 1);

	/* fix indirection block */
	btab.magic1 = 0;
	btab.magic2 = 0;
	btab.ino = inode->i_ino;
	btab.bchecksum = stegfs_chksum(bh->b_data, bh->b_size);
	stegfs_encrypt_btab(inode->i_sb,
			    STEGFS_INO_LVL(inode->i_ino),
			    bh->b_blocknr, &btab);
	stegfs_put_btab(inode->i_sb, bh->b_blocknr, &btab);
	if (cerr)
		*cerr = STEGFS_GB_CREATED;
	if (IS_SYNC(inode) || inode->u.stegfs_i.i_osync) {
		ll_rw_block (WRITE, 1, &bh);
		wait_on_buffer (bh);
	}
	inode->i_ctime = CURRENT_TIME;
	if (ncopy == 0)
		inode->i_blocks += blocks;
	mark_inode_dirty(inode);
	brelse (bh);
	kfree(bufplain);
	return result;
}

#if 0
static struct buffer_head * block_getblk (struct inode * inode,
					  struct buffer_head * bh, int nr,
					  int create, int blocksize, 
					  int new_block, int * err)
{
	return block_getblkn(inode, bh, nr, create, blocksize,
			     new_block, err, 0, 0, NULL);
}
#endif

struct buffer_head * stegfs_getblkn (struct inode * inode, long block,
				     unsigned short * iv,
				     int create, int * err,
				     int ncopy, int recreate, int * cerr)
{
	struct buffer_head *bh, *bh2;
	int offsets[4], *p;
	int depth = stegfs_block_to_path(inode, block, offsets);
	int depth0;
	int i, cerr2 = -1;
	int creating = 0;
	int create2 = 0;
	int ncopy2 = STEGFS_GB_ANYCOPY;
	int ncopy3;
	unsigned short iv2 = 0;
	bh = NULL;

	if (!cerr)
		cerr = &cerr2;

	*err = -EIO;

	if (depth == 0)
		goto fail;

	if (create && STEGFS_IS_HID_INO(inode->i_ino) &&
	    ncopy == STEGFS_GB_ANYCOPY && inode->i_size) {
		if (block < ((inode->i_size-1) >>
			     EXT2_BLOCK_SIZE_BITS(inode->i_sb))+1) {
			create = 0;
		}
	}

	/*
	 * If this is a sequential block allocation, set the next_alloc_block
	 * to this block now so that all the indblock and data block
	 * allocations use the same goal zone
	 */

/*
	stegfs_debug ("block %lu, next %u, goal %u.\n", block, 
		    inode->u.stegfs_i.i_next_alloc_block,
		    inode->u.stegfs_i.i_next_alloc_goal);
*/
	if (block == inode->u.stegfs_i.i_next_alloc_block + 1) {
		inode->u.stegfs_i.i_next_alloc_block++;
		inode->u.stegfs_i.i_next_alloc_goal++;
/*
		if (!STEGFS_IS_HID_INO(inode->i_ino))
			inode->u.stegfs_i.i_next_alloc_goal++;
		else {
			for (i=0; i<STEGFS_MAX_BLOCK_COPIES; i++)
				inode->u.stegfs_i.i_x->i_next_goal_block[i]++;
		}
*/
	}
	/*
	stegfs_debug("depth: %d\n", depth);
	*/
	*err = -ENOSPC;
	if (!STEGFS_IS_HID_INO(inode->i_ino)) {
		/*
		stegfs_debug("non-hid inode\n");
		*/
		bh = inode_getblkn (inode, *(p=offsets), iv, create, block,
				    err, STEGFS_GB_ANYCOPY, &ncopy2,
				    recreate, cerr);
		while (--depth) {
			bh = block_getblkn (inode, bh, *++p, iv, create,
					   inode->i_sb->s_blocksize, block,
					   err, ncopy2, recreate, cerr);
		}
		return bh;
	}

	depth0 = depth;

	if (ncopy != STEGFS_GB_ANYCOPY) {
		/*
		stegfs_debug("hid inode\n");
		*/
		bh = inode_getblkn (inode, *(p=offsets), iv, create, block,
				   err, ncopy, &ncopy2, recreate, cerr);
		while (--depth) {
			create2 = (create || *cerr == STEGFS_GB_RECREATED ||
				   *cerr == STEGFS_GB_CREATED);
			bh = block_getblkn (inode, bh, *++p, iv, create2,
					    inode->i_sb->s_blocksize, block,
					    err, ncopy2, recreate, cerr);
		}
		if (*cerr != STEGFS_GB_CREATED)
			return bh;
		creating = 1;
		inode->u.stegfs_i.i_pref_bcopy = ncopy2;
		brelse(bh);
	}
	/*
	stegfs_debug("hid inode, any copy\n");
	*/
	bh2 = NULL;
	for (i=0; i<inode->u.stegfs_i.i_bcopies; i++) {
		depth = depth0;
		ncopy3 = (i + inode->u.stegfs_i.i_pref_bcopy) % 
			inode->u.stegfs_i.i_bcopies;
		bh = inode_getblkn (inode, *(p=offsets), iv, create, block,
				    err, ncopy3, &ncopy2, recreate, cerr);
		while (--depth) {
			create2 = (create || creating ||
				   *cerr == STEGFS_GB_RECREATED ||
				   *cerr == STEGFS_GB_CREATED);
			bh = block_getblkn (inode, bh, *++p, iv, create2,
					    inode->i_sb->s_blocksize, block,
					    err, ncopy2, recreate, cerr);
		}

		if (*cerr == STEGFS_GB_CREATED || creating) {
			if (bh == NULL) {
				stegfs_debug("null bh\n");
				*err = -EIO;
				return NULL;
			}

			stegfs_debug("allocated ino: %lu block: %lu\n",
				     inode->i_ino,
				     bh->b_blocknr);

			if (i == 0) {
				iv2 = *iv;
				bh2 = bh;
				inode->u.stegfs_i.i_pref_bcopy = 0;
			}
			else
				brelse(bh);
			creating = 1;
			continue;
		}
		if (*cerr != STEGFS_GB_NOVALID) {
			inode->u.stegfs_i.i_pref_bcopy = ncopy3;
			return bh;
		}
	}
	if(bh2 != NULL) {
		*iv = iv2;
		return bh2;
	}
	if (*err != -EFBIG)
		stegfs_warning(inode->i_sb, "stegfs_getblk",
			       "No valid copies of block found"
			       " (inode %lu)", inode->i_ino);
	if(bh) {
		stegfs_debug("brelse\n");
		brelse(bh);
	}
	*cerr = STEGFS_GB_NOVALID;
	if (*err != -EFBIG)
		*err = -EIO;
	return NULL;

	/* return bh; */
fail:
	return NULL;
}

struct buffer_head * stegfs_getblk (struct inode * inode, long block,
				    unsigned short * iv, int create, int * err)
{
	return stegfs_getblkn(inode, block, iv, create, err,
			      STEGFS_GB_ANYCOPY, 0, NULL);
}


struct buffer_head * stegfs_bread (struct inode * inode, int block,
				   unsigned short * iv, int create, int *err)
{
	struct buffer_head * bh;
	int prev_blocks;
	
	prev_blocks = inode->i_blocks;
	
	bh = stegfs_getblk (inode, block, iv, create, err);
	if (!bh)
		return bh;

	/*
	 * If the inode has grown, and this is a directory, then perform
	 * preallocation of a few more blocks to try to keep directory
	 * fragmentation down.
	 */
	if (!STEGFS_IS_HID_INO(inode->i_ino) && create && 
	    S_ISDIR(inode->i_mode) && 
	    inode->i_blocks > prev_blocks &&
	    STEGFS_HAS_COMPAT_FEATURE(inode->i_sb,
				    EXT2_FEATURE_COMPAT_DIR_PREALLOC)) {
		int i;
		struct buffer_head *tmp_bh;
		unsigned short tmp_iv;

		for (i = 1;
		     i < EXT2_SB(inode->i_sb)->s_es->s_prealloc_dir_blocks;
		     i++) {
			/* 
			 * stegfs_getblk will zero out the contents of the
			 * directory for us
			 */
			tmp_bh = stegfs_getblk(inode, block+i,
					       &tmp_iv, create, err);
			if (!tmp_bh) {
				brelse (bh);
				return 0;
			}
			brelse (tmp_bh);
		}
	}
	
	if (buffer_uptodate(bh))
		return bh;
	ll_rw_block (READ, 1, &bh);
	wait_on_buffer (bh);
	if (buffer_uptodate(bh))
		return bh;
	brelse (bh);
	*err = -EIO;
	return NULL;
}

void stegfs_read_inode (struct inode * inode)
{
	struct buffer_head * bh = NULL;
	struct ext2_inode * raw_inode;
	struct stegfs_inode * sraw_inode;
	unsigned long block_group;
	unsigned long group_desc;
	unsigned long desc;
	unsigned long block;
	unsigned long offset;
	struct ext2_group_desc * gdp;

	int slevel,i;
	struct stegfs_btable ibtab;

	int attempts;
#if defined(CONFIG_DIGEST) || defined(CONFIG_DIGEST_MODULE)
	struct digest_implementation *di;
	struct digest_context dx;
#else
	SHA1_CTX sha1ctx;
#endif
	unsigned int digest[5];
	unsigned int keystr[/*4+1+1 = */ 6];
	int success;
	unsigned int maxblock;
	unsigned int scale;

	stegfs_debug("inode num: %lu\n", inode->i_ino);

	inode->u.stegfs_i.i_x = NULL;
	inode->u.stegfs_i.i_x = (struct stegfs_inode_info2 *)
		kmalloc(sizeof(struct stegfs_inode_info2), GFP_KERNEL);
	if (!inode->u.stegfs_i.i_x) {
		printk("StegFS: unable to allocate memory!\n");
		goto bad_inode;
	}
	memset(inode->u.stegfs_i.i_x, 0, sizeof(struct stegfs_inode_info2));

	if (!STEGFS_IS_HID_INO(inode->i_ino)) {
		/* inode from ext2 fs */
		stegfs_debug("getting non-hidden inode\n");
		if ((inode->i_ino != EXT2_ROOT_INO &&
		     inode->i_ino != EXT2_ACL_IDX_INO &&
		     inode->i_ino != EXT2_ACL_DATA_INO &&
		     inode->i_ino < STEGFS_FIRST_INO(inode->i_sb)) ||
		    inode->i_ino >
		    le32_to_cpu(inode->i_sb->u.stegfs_sb.s_es->s_inodes_count)) {
			stegfs_error (inode->i_sb, "stegfs_read_inode",
				      "bad inode number: %lu", inode->i_ino);
			goto bad_inode;
		}
		block_group = (inode->i_ino - 1) / STEGFS_INODES_PER_GROUP(inode->i_sb);
		if (block_group >= inode->i_sb->u.stegfs_sb.s_groups_count) {
			stegfs_error (inode->i_sb, "stegfs_read_inode",
				      "group >= groups count");
			goto bad_inode;
		}
		group_desc = block_group >> STEGFS_DESC_PER_BLOCK_BITS(inode->i_sb);
		desc = block_group & (STEGFS_DESC_PER_BLOCK(inode->i_sb) - 1);
		bh = inode->i_sb->u.stegfs_sb.s_group_desc[group_desc];
		if (!bh) {
			stegfs_error (inode->i_sb, "stegfs_read_inode",
				      "Descriptor not loaded");
			goto bad_inode;
		}
		
		gdp = (struct ext2_group_desc *) bh->b_data;
		/*
		 * Figure out the offset within the block group inode table
		 */
		offset = ((inode->i_ino - 1) % STEGFS_INODES_PER_GROUP(inode->i_sb)) *
			STEGFS_INODE_SIZE(inode->i_sb);
		block = le32_to_cpu(gdp[desc].bg_inode_table) +
			(offset >> EXT2_BLOCK_SIZE_BITS(inode->i_sb));
		if (!(bh = bread (inode->i_dev, block, inode->i_sb->s_blocksize))) {
			stegfs_error (inode->i_sb, "stegfs_read_inode",
				      "unable to read inode block - "
				      "inode=%lu, block=%lu", inode->i_ino, block);
			goto bad_inode;
		}
		offset &= (EXT2_BLOCK_SIZE(inode->i_sb) - 1);
		raw_inode = (struct ext2_inode *) (bh->b_data + offset);

		inode->i_mode = le16_to_cpu(raw_inode->i_mode);
		inode->i_uid = le16_to_cpu(raw_inode->i_uid);
		inode->i_gid = le16_to_cpu(raw_inode->i_gid);
		inode->i_nlink = le16_to_cpu(raw_inode->i_links_count);
		inode->i_size = le32_to_cpu(raw_inode->i_size);
		stegfs_debug("size: %u\n", inode->i_size);
		inode->i_atime = le32_to_cpu(raw_inode->i_atime);
		inode->i_ctime = le32_to_cpu(raw_inode->i_ctime);
		inode->i_mtime = le32_to_cpu(raw_inode->i_mtime);
		inode->u.stegfs_i.i_dtime = le32_to_cpu(raw_inode->i_dtime);
		/* We now have enough fields to check if the inode was active or not.
		 * This is needed because nfsd might try to access dead inodes
		 * the test is that same one that e2fsck uses
		 * NeilBrown 1999oct15
		 */
		if (inode->i_nlink == 0 && (inode->i_mode == 0 ||
					    inode->u.stegfs_i.i_dtime)) {
                        /* this inode is deleted */
			stegfs_debug("this inode is deleted\n");
			brelse (bh);
			goto bad_inode;
		}
		inode->i_blksize = PAGE_SIZE;	/* This is the optimal IO size (for stat), not the fs block size */
		inode->i_blocks = le32_to_cpu(raw_inode->i_blocks);
		inode->i_version = ++global_event;
		inode->i_generation = le32_to_cpu(raw_inode->i_generation);
		inode->u.stegfs_i.i_new_inode = 0;
		inode->u.stegfs_i.i_flags = le32_to_cpu(raw_inode->i_flags);
		inode->u.stegfs_i.i_faddr = le32_to_cpu(raw_inode->i_faddr);
		inode->u.stegfs_i.i_frag_no = raw_inode->i_frag;
		inode->u.stegfs_i.i_frag_size = raw_inode->i_fsize;
		inode->u.stegfs_i.i_osync = 0;
		inode->u.stegfs_i.i_file_acl = le32_to_cpu(raw_inode->i_file_acl);
		if (S_ISDIR(inode->i_mode))
			inode->u.stegfs_i.i_dir_acl = le32_to_cpu(raw_inode->i_dir_acl);
		else {
			inode->u.stegfs_i.i_dir_acl = 0;
			inode->u.stegfs_i.i_high_size =
				le32_to_cpu(raw_inode->i_size_high);
#if BITS_PER_LONG < 64
			if (raw_inode->i_size_high)
				inode->i_size = (__u32)-1;
#else
			inode->i_size |= ((__u64)le32_to_cpu(raw_inode->i_size_high))
				<< 32;
#endif
		}
		inode->u.stegfs_i.i_block_group = block_group;
		inode->u.stegfs_i.i_next_alloc_block = 0;
		inode->u.stegfs_i.i_next_alloc_goal = 0;
		inode->u.stegfs_i.i_pref_bcopy = 0;
		for (block = 0; block < STEGFS_MAX_BLOCK_COPIES; block++) {
			inode->u.stegfs_i.i_x->i_next_goal_block[block] = 0;
		}
		if (inode->u.stegfs_i.i_prealloc_count)
			stegfs_error (inode->i_sb, "stegfs_read_inode",
				      "New inode has non-zero prealloc count!");
		if (S_ISCHR(inode->i_mode) || S_ISBLK(inode->i_mode))
			inode->i_rdev = to_kdev_t(le32_to_cpu(raw_inode->i_block[0]));
		else for (block = 0; block < EXT2_N_BLOCKS; block++)
			inode->u.stegfs_i.i_x->i_data[block] = raw_inode->i_block[block];
					     
		brelse (bh);
		inode->i_op = NULL;
		if (inode->i_ino == EXT2_ACL_IDX_INO ||
		    inode->i_ino == EXT2_ACL_DATA_INO)
			/* Nothing to do */ ;
		else if (S_ISREG(inode->i_mode))
			inode->i_op = &stegfs_file_inode_operations;
		else if (S_ISDIR(inode->i_mode))
			inode->i_op = &stegfs_dir_inode_operations;
		else if (S_ISLNK(inode->i_mode))
			inode->i_op = &stegfs_symlink_inode_operations;
		else if (S_ISCHR(inode->i_mode))
			inode->i_op = &chrdev_inode_operations;
		else if (S_ISBLK(inode->i_mode))
			inode->i_op = &blkdev_inode_operations;
		else if (S_ISFIFO(inode->i_mode))
			init_fifo(inode);
		inode->i_attr_flags = 0;
		if (inode->u.stegfs_i.i_flags & EXT2_SYNC_FL) {
			inode->i_attr_flags |= ATTR_FLAG_SYNCRONOUS;
			inode->i_flags |= MS_SYNCHRONOUS;
		}
		if (inode->u.stegfs_i.i_flags & EXT2_APPEND_FL) {
			inode->i_attr_flags |= ATTR_FLAG_APPEND;
			inode->i_flags |= S_APPEND;
		}
		if (inode->u.stegfs_i.i_flags & EXT2_IMMUTABLE_FL) {
			inode->i_attr_flags |= ATTR_FLAG_IMMUTABLE;
			inode->i_flags |= S_IMMUTABLE;
		}
		if (inode->u.stegfs_i.i_flags & EXT2_NOATIME_FL) {
			inode->i_attr_flags |= ATTR_FLAG_NOATIME;
			inode->i_flags |= MS_NOATIME;
		}
		return;
	}
	/* hidden inode */
	if (inode->i_ino == STEGFS_HID_ROOT_INO) {
		if(!inode->i_sb->u.stegfs_sb.s_x->s_isstegfs)
			goto bad_inode;
		stegfs_debug("getting hidden root inode\n");
		/* magic up an inode for the root dir of the
		   hidden part of the fs */
		inode->i_mode = S_IFDIR|S_IRUSR|S_IXUSR|S_IRGRP|
			S_IXGRP|S_IROTH|S_IXOTH;
		inode->i_uid = STEGFS_SB(inode->i_sb)->s_resuid;
		inode->i_gid = STEGFS_SB(inode->i_sb)->s_resgid;
		inode->i_nlink = 2 /*+ inode->i_sb->u.stegfs_sb.s_seclevel*/;
		inode->i_size = EXT2_BLOCK_SIZE(inode->i_sb);
		inode->i_atime = CURRENT_TIME;
		inode->i_ctime = CURRENT_TIME;
		inode->i_mtime = CURRENT_TIME;
		inode->u.stegfs_i.i_dtime = 0;
		inode->i_blksize = PAGE_SIZE;
		inode->i_blocks = 1;
		inode->i_version = ++global_event;
		inode->i_generation = 0;
		inode->u.stegfs_i.i_new_inode = 0;
		inode->u.stegfs_i.i_flags = 0;
		inode->u.stegfs_i.i_faddr = 0;
		inode->u.stegfs_i.i_frag_no = 0;
		inode->u.stegfs_i.i_frag_size = 0;
		inode->u.stegfs_i.i_osync = 0;
		inode->u.stegfs_i.i_file_acl = 0;
		if (S_ISDIR(inode->i_mode))
			inode->u.stegfs_i.i_dir_acl = 0;
		else {
			inode->u.stegfs_i.i_dir_acl = 0;
			inode->u.stegfs_i.i_high_size = 0;

		}
		inode->u.stegfs_i.i_block_group = 0; /* FIXME: CHECK */
		inode->u.stegfs_i.i_next_alloc_block = 0;
		inode->u.stegfs_i.i_next_alloc_goal = 0;
		inode->u.stegfs_i.i_pref_bcopy = 0;
		for (block = 0; block < STEGFS_MAX_BLOCK_COPIES; block++) {
			inode->u.stegfs_i.i_x->i_next_goal_block[block] = 0;
		}
		for (block = 0; block < EXT2_N_BLOCKS; block++)
			inode->u.stegfs_i.i_x->i_data[block] = 0;

		inode->i_op = &stegfs_dir_inode_operations;

		inode->i_attr_flags = 0;
		inode->i_attr_flags |= ATTR_FLAG_NOATIME;
		inode->i_flags |= MS_NOATIME;
		/* for stegfs inodes */
		inode->u.stegfs_i.i_icopies = 0;
		inode->u.stegfs_i.i_bcopies = 0;
		return;
                       
	}

	/* normal hidden inode */

	stegfs_debug("getting normal hidden inode\n");
	/* find the entry for it in the list */
	slevel = STEGFS_INO_LVL(inode->i_ino);
#if defined(CONFIG_DIGEST) || defined(CONFIG_DIGEST_MODULE)
	di = STEGFS_SB(inode->i_sb)->s_x->s_di;
	dx.di = di;
#endif
	if(!inode->i_sb->u.stegfs_sb.s_x->s_slkeys[slevel-1])
		goto bad_inode;

	maxblock = STEGFS_SB(inode->i_sb)->s_es->s_blocks_count - 1;
	scale = ((unsigned int)-1) / maxblock;

	memcpy((unsigned char *)keystr,
	       inode->i_sb->u.stegfs_sb.s_x->s_slkeys[slevel-1], 16);
	keystr[4] = inode->i_ino;
	keystr[5] = 0;
#if defined(CONFIG_DIGEST) || defined(CONFIG_DIGEST_MODULE)
	dx.digest_info = (u32 *)kmalloc(di->working_size, GFP_KERNEL);
	di->open(&dx);
	di->update(&dx, (unsigned char *)keystr, 24);
	di->close(&dx, (unsigned char *)digest);
	memset(dx.digest_info, 0, di->working_size);
	kfree(dx.digest_info);
#else
	SHA1Init(&sha1ctx);
	SHA1Update(&sha1ctx, (unsigned char *)keystr, 24);
	SHA1Final((unsigned char *)digest, &sha1ctx);
#endif
	attempts=0;
	success=0;
	block = -1;
	while (1) {

		for (i=0; i<5; i++) {
			block = digest[i] / scale;
#ifdef STEGFS_DEBUG
			if (attempts < 50) {
				stegfs_debug("trying: %lu\n", block);
			}
			else if(attempts == 50) {
				stegfs_debug("trying: more (not logging)\n");
			}
#endif
			if (!(block > maxblock || block < 1)) {
				/* read block */
				if (!(bh = bread (inode->i_dev, block,
						  inode->i_sb->s_blocksize))) {
					stegfs_error (inode->i_sb,
						      "stegfs_read_inode",
						      "unable to read block - "
						      "inode=%lu, block=%lu",
						      inode->i_ino, block);
					goto bad_inode;
				}

				stegfs_get_btab(inode->i_sb, block, &ibtab);
				stegfs_decrypt_btab(inode->i_sb, slevel,
						    block, &ibtab);
				if(ibtab.magic1 == 0 && ibtab.magic2 == 1 &&
				   stegfs_chksum(bh->b_data, bh->b_size)
				   == le32_to_cpu(ibtab.bchecksum)) {
					success = 1;
				}
				else
					bforget(bh);
			}
			if(success)
				break;
			attempts++;
			if(attempts > STEGFS_INO_BATT)
				goto bad_inode;
		}

		if(success)
			break;

#if defined(CONFIG_DIGEST) || defined(CONFIG_DIGEST_MODULE)
		dx.digest_info = (u32 *)kmalloc(di->working_size, GFP_KERNEL);
		di->open(&dx);
		keystr[5]++;
		di->update(&dx, (unsigned char *)keystr, 24);
		di->close(&dx, (unsigned char *)digest);
		memset(dx.digest_info, 0, di->working_size);
		kfree(dx.digest_info);
#else
		SHA1Init(&sha1ctx);
		keystr[5]++;
		SHA1Update(&sha1ctx, (unsigned char *)keystr, 24);
		SHA1Final((unsigned char *)digest, &sha1ctx);
#endif		
	}

	/* got a valid ino */
	sraw_inode = (struct stegfs_inode *)
		kmalloc(sizeof(struct stegfs_inode), GFP_KERNEL);
#ifdef STEGFS_DEBUG
	if (sraw_inode == NULL)
		stegfs_debug("kmalloc failed\n");
#endif
	stegfs_decrypt_cbc2(inode->i_sb, slevel, block,
			    bh->b_data, (char *)sraw_inode,
			    sizeof(struct stegfs_inode), ibtab.iv);

	inode->i_mode = le16_to_cpu(sraw_inode->i_mode);
	inode->i_uid = le16_to_cpu(sraw_inode->i_uid);
	inode->i_gid = le16_to_cpu(sraw_inode->i_gid);
	inode->i_nlink = le16_to_cpu(sraw_inode->i_links_count);
	inode->i_size = le32_to_cpu(sraw_inode->i_size);
	inode->i_atime = le32_to_cpu(sraw_inode->i_atime);
	inode->i_ctime = le32_to_cpu(sraw_inode->i_ctime);
	inode->i_mtime = le32_to_cpu(sraw_inode->i_mtime);
	inode->u.stegfs_i.i_dtime = le32_to_cpu(sraw_inode->i_dtime);
	/* actually, this next check probably doesn't make any sense
	 * in the context of StegFS hidden inodes since they are
	 * overwritten on deletion.
	 */
	if (inode->i_nlink == 0 && (inode->i_mode == 0 ||
				    inode->u.stegfs_i.i_dtime)) {
		/* this inode is deleted */
		brelse (bh);
		goto bad_inode;
	}
	inode->i_blksize = PAGE_SIZE;
	/* This is the optimal IO size (for stat), not the fs block size */

	inode->i_blocks = le32_to_cpu(sraw_inode->i_blocks);
	inode->i_version = ++global_event;
	inode->i_generation = le32_to_cpu(sraw_inode->i_generation);
	inode->u.stegfs_i.i_new_inode = 0;
	inode->u.stegfs_i.i_flags = le32_to_cpu(sraw_inode->i_flags);
	inode->u.stegfs_i.i_faddr = le32_to_cpu(sraw_inode->i_faddr);
	inode->u.stegfs_i.i_frag_no = sraw_inode->i_frag;
	inode->u.stegfs_i.i_frag_size = sraw_inode->i_fsize;
	inode->u.stegfs_i.i_osync = 0;
	inode->u.stegfs_i.i_file_acl = le32_to_cpu(sraw_inode->i_file_acl);
	if (S_ISDIR(inode->i_mode))
		inode->u.stegfs_i.i_dir_acl =
			le32_to_cpu(sraw_inode->i_dir_acl);
	else {
		inode->u.stegfs_i.i_dir_acl = 0;
		inode->u.stegfs_i.i_high_size =
			le32_to_cpu(sraw_inode->i_size_high);
#if BITS_PER_LONG < 64
		if (sraw_inode->i_size_high)
			inode->i_size = (__u32)-1;
#else
		inode->i_size |=
			((__u64)le32_to_cpu(sraw_inode->i_size_high)) << 32;
#endif
	}
	inode->u.stegfs_i.i_block_group = 0;
	inode->u.stegfs_i.i_next_alloc_block = 0;
	inode->u.stegfs_i.i_next_alloc_goal = 0;
	inode->u.stegfs_i.i_pref_bcopy = 0;
	for (block = 0; block < STEGFS_MAX_BLOCK_COPIES; block++) {
		inode->u.stegfs_i.i_x->i_next_goal_block[block] = 0;
	}
	if (inode->u.stegfs_i.i_prealloc_count)
		stegfs_error (inode->i_sb,
			      "stegfs_read_inode",
			      "New inode has non-zero"
			      "prealloc count!");
	if (S_ISCHR(inode->i_mode) || S_ISBLK(inode->i_mode))
		inode->i_rdev = to_kdev_t(le32_to_cpu(sraw_inode->i_block[0]));
	else for (block = 0; block < EXT2_N_BLOCKS *
			  STEGFS_MAX_BLOCK_COPIES; block++)
		inode->u.stegfs_i.i_x->i_data[block] = sraw_inode->i_block[block];

	inode->u.stegfs_i.i_icopies = le32_to_cpu(sraw_inode->i_icopies);
	inode->u.stegfs_i.i_bcopies = le32_to_cpu(sraw_inode->i_bcopies);
	for (block = 0; block < STEGFS_MAX_INO_COPIES; block++)
		inode->u.stegfs_i.i_x->i_inode[block] =
			le32_to_cpu(sraw_inode->i_inode[block]);
	brelse (bh);
	inode->i_op = NULL;
	if (inode->i_ino == EXT2_ACL_IDX_INO ||
	    inode->i_ino == EXT2_ACL_DATA_INO)
             /* Nothing to do */ ;
	else if (S_ISREG(inode->i_mode))
		inode->i_op = &stegfs_file_inode_operations;
	else if (S_ISDIR(inode->i_mode))
		inode->i_op = &stegfs_dir_inode_operations;
	else if (S_ISLNK(inode->i_mode))
		inode->i_op = &stegfs_symlink_inode_operations;
	else if (S_ISCHR(inode->i_mode))
		inode->i_op = &chrdev_inode_operations;
	else if (S_ISBLK(inode->i_mode))
		inode->i_op = &blkdev_inode_operations;
	else if (S_ISFIFO(inode->i_mode))
		init_fifo(inode);
	inode->i_attr_flags = 0;
	if (inode->u.stegfs_i.i_flags & EXT2_SYNC_FL) {
		inode->i_attr_flags |= ATTR_FLAG_SYNCRONOUS;
		inode->i_flags |= MS_SYNCHRONOUS;
	}
	if (inode->u.stegfs_i.i_flags & EXT2_APPEND_FL) {
		inode->i_attr_flags |= ATTR_FLAG_APPEND;
		inode->i_flags |= S_APPEND;
	}
	if (inode->u.stegfs_i.i_flags & EXT2_IMMUTABLE_FL) {
		inode->i_attr_flags |= ATTR_FLAG_IMMUTABLE;
		inode->i_flags |= S_IMMUTABLE;
	}
	if (inode->u.stegfs_i.i_flags & EXT2_NOATIME_FL) {
		inode->i_attr_flags |= ATTR_FLAG_NOATIME;
		inode->i_flags |= MS_NOATIME;
	}

	kfree(sraw_inode);
	return;

bad_inode:
	stegfs_debug("bad inode\n");
	kfree(inode->u.stegfs_i.i_x);
	make_bad_inode(inode);
	return;
}

int stegfs_blkused(struct super_block *sb, int bnum);

static int stegfs_update_inode(struct inode * inode, int do_sync)
{
	struct buffer_head * bh;
	struct ext2_inode * raw_inode;
	struct stegfs_inode * sraw_inode;
	struct stegfs_btable btab, btab2;
	unsigned long block_group;
	unsigned long group_desc;
	unsigned long desc;
	unsigned long block = 0;
	unsigned long offset;
	int err = 0;
	int i,j,k;
	int slevel, recreate = 0;
	struct ext2_group_desc * gdp;

	stegfs_debug("ino: %lu\n", inode->i_ino);
	if(!STEGFS_IS_HID_INO(inode->i_ino)) {

		if ((inode->i_ino != EXT2_ROOT_INO &&
		     inode->i_ino < STEGFS_FIRST_INO(inode->i_sb)) ||
		    inode->i_ino > le32_to_cpu(inode->i_sb->u.stegfs_sb.s_es->s_inodes_count)) {
			stegfs_error (inode->i_sb, "stegfs_write_inode",
				      "bad inode number: %lu", inode->i_ino);
			return -EIO;
		}
		block_group = (inode->i_ino - 1) / STEGFS_INODES_PER_GROUP(inode->i_sb);
		if (block_group >= inode->i_sb->u.stegfs_sb.s_groups_count) {
			stegfs_error (inode->i_sb, "stegfs_write_inode",
				      "group >= groups count");
			return -EIO;
		}
		group_desc = block_group >> STEGFS_DESC_PER_BLOCK_BITS(inode->i_sb);
		desc = block_group & (STEGFS_DESC_PER_BLOCK(inode->i_sb) - 1);
		bh = inode->i_sb->u.stegfs_sb.s_group_desc[group_desc];
		if (!bh) {
			stegfs_error (inode->i_sb, "stegfs_write_inode",
				      "Descriptor not loaded");
			return -EIO;
		}
		gdp = (struct ext2_group_desc *) bh->b_data;
		/*
		 * Figure out the offset within the block group inode table
		 */
		offset = ((inode->i_ino - 1) % STEGFS_INODES_PER_GROUP(inode->i_sb)) *
			STEGFS_INODE_SIZE(inode->i_sb);
		block = le32_to_cpu(gdp[desc].bg_inode_table) +
			(offset >> EXT2_BLOCK_SIZE_BITS(inode->i_sb));
		if (!(bh = bread (inode->i_dev, block, inode->i_sb->s_blocksize))) {
			stegfs_error (inode->i_sb, "stegfs_write_inode",
				      "unable to read inode block - "
				      "inode=%lu, block=%lu", inode->i_ino, block);
			return -EIO;
		}
		offset &= EXT2_BLOCK_SIZE(inode->i_sb) - 1;
		raw_inode = (struct ext2_inode *) (bh->b_data + offset);
		
		raw_inode->i_mode = cpu_to_le16(inode->i_mode);
		raw_inode->i_uid = cpu_to_le16(inode->i_uid);
		raw_inode->i_gid = cpu_to_le16(inode->i_gid);
		raw_inode->i_links_count = cpu_to_le16(inode->i_nlink);
		raw_inode->i_size = cpu_to_le32(inode->i_size);
		raw_inode->i_atime = cpu_to_le32(inode->i_atime);
		raw_inode->i_ctime = cpu_to_le32(inode->i_ctime);
		raw_inode->i_mtime = cpu_to_le32(inode->i_mtime);
		raw_inode->i_blocks = cpu_to_le32(inode->i_blocks);
		raw_inode->i_generation = cpu_to_le32(inode->i_generation);
 		raw_inode->i_dtime = cpu_to_le32(inode->u.stegfs_i.i_dtime);
		raw_inode->i_flags = cpu_to_le32(inode->u.stegfs_i.i_flags);
		raw_inode->i_faddr = cpu_to_le32(inode->u.stegfs_i.i_faddr);
		raw_inode->i_frag = inode->u.stegfs_i.i_frag_no;
		raw_inode->i_fsize = inode->u.stegfs_i.i_frag_size;
		raw_inode->i_file_acl = cpu_to_le32(inode->u.stegfs_i.i_file_acl);
		if (S_ISDIR(inode->i_mode))
			raw_inode->i_dir_acl = cpu_to_le32(inode->u.stegfs_i.i_dir_acl);
		else { 
#if BITS_PER_LONG < 64
			raw_inode->i_size_high =
				cpu_to_le32(inode->u.stegfs_i.i_high_size);
#else
			raw_inode->i_size_high = cpu_to_le32(inode->i_size >> 32);
#endif
		}
		if (S_ISCHR(inode->i_mode) || S_ISBLK(inode->i_mode))
			raw_inode->i_block[0] = cpu_to_le32(kdev_t_to_nr(inode->i_rdev));
		else for (block = 0; block < EXT2_N_BLOCKS; block++)
			raw_inode->i_block[block] = inode->u.stegfs_i.i_x->i_data[block];
		mark_buffer_dirty(bh, 1);
		if (do_sync) {
			ll_rw_block (WRITE, 1, &bh);
			wait_on_buffer (bh);
			if (buffer_req(bh) && !buffer_uptodate(bh)) {
				printk ("IO error syncing ext2 inode ["
					"%s:%08lx]\n",
					bdevname(inode->i_dev), inode->i_ino);
				err = -EIO;
			}
		}
		brelse (bh);
		return err;
	}
	else {
		unsigned int tryblocks[5];
		int btotry = 99;
#if defined(CONFIG_DIGEST) || defined(CONFIG_DIGEST_MODULE)
		struct digest_implementation *di;
		struct digest_context dx;
#else
		SHA1_CTX sha1ctx;
#endif
		unsigned int keystr[6];
		unsigned int randbsx;
		unsigned int maxblock;
		unsigned int scale;
		char *inode1;

		slevel = STEGFS_INO_LVL(inode->i_ino);
#if defined(CONFIG_DIGEST) || defined(CONFIG_DIGEST_MODULE)
		di = STEGFS_SB(inode->i_sb)->s_x->s_di;
		dx.di = di;
#endif
		if (!inode->i_sb->u.stegfs_sb.s_x->s_slkeys[slevel-1]) {
                        stegfs_error (inode->i_sb, "stegfs_update_inode",
				      "cannot write inode %lu, level closed",
				      inode->i_ino);
			return -EIO;
		}
		maxblock = STEGFS_SB(inode->i_sb)->s_es->s_blocks_count - 1;
		scale = ((unsigned int)-1) / maxblock;

		for(i=0; i<inode->u.stegfs_i.i_icopies; i++) {
#if 0
			stegfs_debug("ino: %lu copy: %d\n", inode->i_ino, i);
#endif
			/* Hmm. This could probably all be done less times. */
			/* ought to check that the block hasn't been reused */
			if (!(bh = bread (inode->i_dev,
					  inode->u.stegfs_i.i_x->i_inode[i],
					  inode->i_sb->s_blocksize))) {
				stegfs_error (inode->i_sb, "stegfs_write_inode",
					      "unable to read inode block - "
					      "inode=%lu, block=%lu",
					      inode->i_ino, block);
				return -EIO;
			}
			if (!buffer_uptodate(bh)) {
				ll_rw_block (READ, 1, &bh);
				wait_on_buffer (bh);
			}
			if (!buffer_uptodate(bh)) {
				brelse(bh);
				return -EIO;
			}
			stegfs_get_btab(inode->i_sb,
					inode->u.stegfs_i.i_x->i_inode[i], &btab);
			stegfs_decrypt_btab(inode->i_sb,
					    STEGFS_INO_LVL(inode->i_ino),
					    inode->u.stegfs_i.i_x->i_inode[i], &btab);
			if(btab.magic1 != 0 || btab.magic2 != 1 ||
			   btab.ino != inode->i_ino ||
			   stegfs_chksum(bh->b_data, bh->b_size)
			   != le32_to_cpu(btab.bchecksum)) {
				/* realloc copy */
				recreate = 1;
				if (btotry==99) {
					memcpy((unsigned char *)keystr,
					       inode->i_sb->
					       u.stegfs_sb.s_x->s_slkeys[slevel-1],
					       16);
					keystr[4] = inode->i_ino;
					keystr[5] = 0;
#if defined(CONFIG_DIGEST) || defined(CONFIG_DIGEST_MODULE)
					dx.digest_info =
					  (u32 *)kmalloc(di->working_size,
							 GFP_KERNEL);
					di->open(&dx);
					di->update(&dx,
						   (unsigned char *)keystr,
						   24);
					di->close(&dx,
						  (unsigned char *)tryblocks);
					memset(dx.digest_info, 0,
					       dx.di->working_size);
					kfree(dx.digest_info);
#else
					SHA1Init(&sha1ctx);
					SHA1Update(&sha1ctx,
						   (unsigned char *)keystr,
						   24);
					SHA1Final((unsigned char *)tryblocks,
						  &sha1ctx);
#endif
					btotry = 0;
				}

				k=0;
				inode->u.stegfs_i.i_x->i_inode[i] = 0;

				while (1) {

					for (j=btotry; j<5; j++) {

						randbsx = tryblocks[j] / scale;

						if (!(randbsx > maxblock || randbsx < 1 ||
						      stegfs_blkused(inode->i_sb,randbsx))) {

							inode->u.stegfs_i.i_x->i_inode[i] = randbsx;
						}
						if (inode->u.stegfs_i.i_x->i_inode[i])
							break;
					}
					if (inode->u.stegfs_i.i_x->i_inode[i]) {
						btotry = j+1;
						break;
					}
					k++;
					if (k > STEGFS_INO_BATT)
						break;
					btotry = 0;
#if defined(CONFIG_DIGEST) || defined(CONFIG_DIGEST_MODULE)
					dx.digest_info =
					  (u32 *)kmalloc(di->working_size,
							 GFP_KERNEL);
					di->open(&dx);
					keystr[5]++;
					di->update(&dx, 
						   (unsigned char *)keystr,
						   24);
					di->close(&dx,
						  (unsigned char *)tryblocks);
					memset(dx.digest_info, 0,
					       dx.di->working_size);
					kfree(dx.digest_info);
#else
					SHA1Init(&sha1ctx);
					keystr[5]++;
					SHA1Update(&sha1ctx,
						   (unsigned char *)keystr,
						   24);
					SHA1Final((unsigned char *)tryblocks, &sha1ctx);
#endif

				}

				if (!inode->u.stegfs_i.i_x->i_inode[i]) {
					stegfs_debug("Bleurgh. Can't "
					       "find free block for inode\n");
					return -EIO;
					/* FIXME: this error? */
				}

				stegfs_debug("using block: %u\n",
					     inode->u.stegfs_i.i_x->i_inode[i]);

				stegfs_get_btab(inode->i_sb,
						bh->b_blocknr, &btab);
				k = 0;
				for (j=0; j<(STEGFS_MAX_LEVELS-1); j++) {
					if (STEGFS_SB(inode->i_sb)->s_x->s_slkeys[j]) {
						stegfs_decrypt_btab2(inode->i_sb,
								     j+1, bh->b_blocknr,
								     &btab, &btab2);
						if (btab2.magic1 == 0 &&
						    (btab2.magic2 == 0 || btab2.magic2 == 1) &&
						    btab2.ino != inode->i_ino) {
							k = 1;
							break;
						}
					}
				}
				if (!k) {
					get_random_bytes(&btab,
							 sizeof(struct stegfs_btable));
					stegfs_put_btab(inode->i_sb,
							bh->b_blocknr, &btab);
				}

				brelse(bh);

				if (!(bh = bread (inode->i_dev,
						  inode->u.stegfs_i.i_x->i_inode[i],
						  inode->i_sb->s_blocksize))) {
					stegfs_error (inode->i_sb, "stegfs_write_inode",
						      "unable to read inode block - "
						      "inode=%lu, block=%lu",
						      inode->i_ino, block);
					return -EIO;
				}
				btab.iv = -1;
			}
			sraw_inode = (struct stegfs_inode *)
				kmalloc(sizeof(struct stegfs_inode), GFP_KERNEL);
#ifdef STEGFS_DEBUG
			if (sraw_inode == NULL)
				stegfs_debug("kmalloc failed\n");
#endif
			sraw_inode->i_mode = cpu_to_le16(inode->i_mode);
			sraw_inode->i_uid = cpu_to_le16(inode->i_uid);
			sraw_inode->i_gid = cpu_to_le16(inode->i_gid);
			sraw_inode->i_links_count = cpu_to_le16(inode->i_nlink);
			sraw_inode->i_size = cpu_to_le32(inode->i_size);
			sraw_inode->i_atime = cpu_to_le32(inode->i_atime);
			sraw_inode->i_ctime = cpu_to_le32(inode->i_ctime);
			sraw_inode->i_mtime = cpu_to_le32(inode->i_mtime);
			sraw_inode->i_blocks = cpu_to_le32(inode->i_blocks);
			sraw_inode->i_generation = cpu_to_le32(inode->i_generation);
			sraw_inode->i_dtime = cpu_to_le32(inode->
							 u.stegfs_i.i_dtime);
			sraw_inode->i_flags = cpu_to_le32(inode->
							 u.stegfs_i.i_flags);
			sraw_inode->i_faddr = cpu_to_le32(inode->
							 u.stegfs_i.i_faddr);
			sraw_inode->i_frag = inode->u.stegfs_i.i_frag_no;
			sraw_inode->i_fsize = inode->u.stegfs_i.i_frag_size;
			sraw_inode->i_file_acl = cpu_to_le32(inode->
							    u.stegfs_i.i_file_acl);

			if (S_ISDIR(inode->i_mode))
				sraw_inode->i_dir_acl =
					cpu_to_le32(inode->u.stegfs_i.i_dir_acl);
			else { 
#if BITS_PER_LONG < 64
				sraw_inode->i_size_high =
					cpu_to_le32(inode->u.stegfs_i.i_high_size);
#else
				sraw_inode->i_size_high = cpu_to_le32(inode->i_size >> 32);
#endif
			}
			if (S_ISCHR(inode->i_mode) || S_ISBLK(inode->i_mode))
				sraw_inode->i_block[0] =
					cpu_to_le32(kdev_t_to_nr(inode->i_rdev));
			else for (block = 0; block < EXT2_N_BLOCKS
					  * STEGFS_MAX_BLOCK_COPIES; block++) {
				sraw_inode->i_block[block] =
				  inode->u.stegfs_i.i_x->i_data[block];
			}

			for (block = 0; block < STEGFS_MAX_INO_COPIES;
			     block++) {
				sraw_inode->i_inode[block] =
					cpu_to_le32(inode->
						    u.stegfs_i.i_x->i_inode[block]);
			}

			sraw_inode->i_icopies = cpu_to_le32(inode->
							   u.stegfs_i.i_icopies);
			sraw_inode->i_bcopies = cpu_to_le32(inode->
							   u.stegfs_i.i_bcopies);

			inode1 = kmalloc(inode->i_sb->s_blocksize, GFP_KERNEL);
			memset(inode1, 0, inode->i_sb->s_blocksize);
			memcpy(inode1, sraw_inode, sizeof(struct stegfs_inode));
			stegfs_encrypt_cbc2(inode->i_sb, slevel,
					    bh->b_blocknr,
					    (char *)inode1, bh->b_data,
					    inode->i_sb->s_blocksize,
					    &btab.iv);

			mark_buffer_dirty(bh, 1);

			btab.magic1 = 0;
			btab.magic2 = 1;
			btab.ino = inode->i_ino;

			btab.bchecksum = stegfs_chksum(bh->b_data,
						       bh->b_size);
			stegfs_encrypt_btab(inode->i_sb,
					    STEGFS_INO_LVL(inode->i_ino),
					    inode->u.stegfs_i.i_x->i_inode[i], &btab);
			stegfs_put_btab(inode->i_sb,
					inode->u.stegfs_i.i_x->i_inode[i],
					&btab);

			if (do_sync) {
				ll_rw_block (WRITE, 1, &bh);
				wait_on_buffer (bh);
				if (buffer_req(bh) && !buffer_uptodate(bh)) {
					printk ("IO error syncing ext2 inode ["
						"%s:%08lx]\n",
						bdevname(inode->i_dev),
						inode->i_ino);
					err = -EIO;
				}

			}

			kfree(inode1);
			brelse (bh);
			kfree(sraw_inode);
			if(err!=0)
				return err;
		}

		if (!recreate)
			return err;
		return stegfs_update_inode(inode, do_sync);

	}
}

void stegfs_write_inode (struct inode * inode)
{
	stegfs_update_inode (inode, 0);
}

int stegfs_sync_inode (struct inode *inode)
{
	return stegfs_update_inode (inode, 1);
}

int stegfs_notify_change(struct dentry *dentry, struct iattr *iattr)
{
	struct inode *inode = dentry->d_inode;
	int		retval;
	unsigned int	flags;
	
	retval = -EPERM;
	if (iattr->ia_valid & ATTR_ATTR_FLAG)
	{
		if((!(iattr->ia_attr_flags & ATTR_FLAG_APPEND) !=
		    !(inode->u.stegfs_i.i_flags & EXT2_APPEND_FL)) ||
		   (!(iattr->ia_attr_flags & ATTR_FLAG_IMMUTABLE) !=
		    !(inode->u.stegfs_i.i_flags & EXT2_IMMUTABLE_FL))) {
			if (!capable(CAP_LINUX_IMMUTABLE))
				goto out;
		} else if ((current->fsuid != inode->i_uid) &&
			   !capable(CAP_FOWNER))
			goto out;
	}

	if (iattr->ia_valid & ATTR_SIZE) {
		off_t size = iattr->ia_size;
		unsigned long limit = current->rlim[RLIMIT_FSIZE].rlim_cur;

		if (size < 0)
			return -EINVAL;
#if BITS_PER_LONG == 64
		if (size > ext2_max_sizes[EXT2_BLOCK_SIZE_BITS(inode->i_sb)])
			return -EFBIG;
#endif
		if (limit < RLIM_INFINITY && size > limit) {
			send_sig(SIGXFSZ, current, 0);
			return -EFBIG;
		}

#if BITS_PER_LONG == 64
		if (size >> 33) {
			struct super_block *sb = inode->i_sb;
			struct ext2_super_block *es = sb->u.stegfs_sb.s_es;
			if (!(es->s_feature_ro_compat &
			      cpu_to_le32(EXT2_FEATURE_RO_COMPAT_LARGE_FILE))){
				/* If this is the first large file
				 * created, add a flag to the superblock */
				es->s_feature_ro_compat |=
				cpu_to_le32(EXT2_FEATURE_RO_COMPAT_LARGE_FILE);
				mark_buffer_dirty(sb->u.stegfs_sb.s_sbh, 1);
			}
		}
#endif
	}

	retval = inode_change_ok(inode, iattr);
	if (retval != 0)
		goto out;

	inode_setattr(inode, iattr);
	
	if (iattr->ia_valid & ATTR_ATTR_FLAG) {
		flags = iattr->ia_attr_flags;
		if (flags & ATTR_FLAG_SYNCRONOUS) {
			inode->i_flags |= MS_SYNCHRONOUS;
			inode->u.stegfs_i.i_flags |= EXT2_SYNC_FL;
		} else {
			inode->i_flags &= ~MS_SYNCHRONOUS;
			inode->u.stegfs_i.i_flags &= ~EXT2_SYNC_FL;
		}
		if (flags & ATTR_FLAG_NOATIME) {
			inode->i_flags |= MS_NOATIME;
			inode->u.stegfs_i.i_flags |= EXT2_NOATIME_FL;
		} else {
			inode->i_flags &= ~MS_NOATIME;
			inode->u.stegfs_i.i_flags &= ~EXT2_NOATIME_FL;
		}
		if (flags & ATTR_FLAG_APPEND) {
			inode->i_flags |= S_APPEND;
			inode->u.stegfs_i.i_flags |= EXT2_APPEND_FL;
		} else {
			inode->i_flags &= ~S_APPEND;
			inode->u.stegfs_i.i_flags &= ~EXT2_APPEND_FL;
		}
		if (flags & ATTR_FLAG_IMMUTABLE) {
			inode->i_flags |= S_IMMUTABLE;
			inode->u.stegfs_i.i_flags |= EXT2_IMMUTABLE_FL;
		} else {
			inode->i_flags &= ~S_IMMUTABLE;
			inode->u.stegfs_i.i_flags &= ~EXT2_IMMUTABLE_FL;
		}
	}
	mark_inode_dirty(inode);
out:
	return retval;
}

