/*
 *  linux/fs/stegfs/dir.c
 *
 * Copyright (C) 1999-2001
 * Andrew McDonald (andrew@mcdonald.org.uk)
 *
 *  from
 *
 *  linux/fs/ext2/dir.c
 *
 *  Copyright (C) 1992, 1993, 1994, 1995
 *  Remy Card (card@masi.ibp.fr)
 *  Laboratoire MASI - Institut Blaise Pascal
 *  Universite Pierre et Marie Curie (Paris VI)
 *
 *  from
 *
 *  linux/fs/minix/dir.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 *
 *  ext2 directory handling functions
 *
 *  Big-endian to little-endian byte-swapping/bitmaps by
 *        David S. Miller (davem@caip.rutgers.edu), 1995
 */

#include "fs.h"
#include "stegfs_fs.h"

#include <asm/uaccess.h>

#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/stat.h>
#include <linux/slab.h>

static ssize_t stegfs_dir_read (struct file * filp, char * buf,
			      size_t count, loff_t *ppos)
{
	return -EISDIR;
}

static int stegfs_readdir(struct file *, void *, filldir_t);

static struct file_operations stegfs_dir_operations = {
	NULL,			/* lseek - default */
	stegfs_dir_read,		/* read */
	NULL,			/* write - bad */
	stegfs_readdir,		/* readdir */
	NULL,			/* poll - default */
	stegfs_ioctl,		/* ioctl */
	NULL,			/* mmap */
	NULL,			/* no special open code */
	NULL,			/* flush */
	NULL,			/* no special release code */
	stegfs_sync_file,		/* fsync */
	NULL,			/* fasync */
	NULL,			/* check_media_change */
	NULL			/* revalidate */
};

/*
 * directories can handle most operations...
 */
struct inode_operations stegfs_dir_inode_operations = {
	&stegfs_dir_operations,	/* default directory file-ops */
	stegfs_create,		/* create */
	stegfs_lookup,		/* lookup */
	stegfs_link,		/* link */
	stegfs_unlink,		/* unlink */
	stegfs_symlink,		/* symlink */
	stegfs_mkdir,		/* mkdir */
	stegfs_rmdir,		/* rmdir */
	stegfs_mknod,		/* mknod */
	stegfs_rename,		/* rename */
	NULL,			/* readlink */
	NULL,			/* follow_link */
	NULL,			/* readpage */
	NULL,			/* writepage */
	NULL,			/* bmap */
	NULL,			/* truncate */
	stegfs_permission,	/* permission */
	NULL			/* smap */
};

int stegfs_check_dir_entry (const char * function, struct inode * dir,
			  struct ext2_dir_entry_2 * de,
			  char * b_data,
			  unsigned long offset)
{
	const char * error_msg = NULL;

	if (le16_to_cpu(de->rec_len) < EXT2_DIR_REC_LEN(1))
		error_msg = "rec_len is smaller than minimal";
	else if (le16_to_cpu(de->rec_len) % 4 != 0)
		error_msg = "rec_len % 4 != 0";
	else if (le16_to_cpu(de->rec_len) < EXT2_DIR_REC_LEN(de->name_len))
		error_msg = "rec_len is too small for name_len";
	else if (dir && ((char *) de - b_data) + le16_to_cpu(de->rec_len) >
		 dir->i_sb->s_blocksize)
		error_msg = "directory entry across blocks";
	else if (dir && le32_to_cpu(de->inode) > le32_to_cpu(dir->i_sb->u.stegfs_sb.s_es->s_inodes_count) && !STEGFS_IS_HID_INO(le32_to_cpu(de->inode)))
		error_msg = "inode out of bounds";

	if (error_msg != NULL)
		stegfs_error (dir->i_sb, function, "bad entry in directory #%lu: %s - "
			    "offset=%lu, inode=%lu, rec_len=%d, name_len=%d",
			    dir->i_ino, error_msg, offset,
			    (unsigned long) le32_to_cpu(de->inode),
			    le16_to_cpu(de->rec_len), de->name_len);
	return error_msg == NULL ? 1 : 0;
}

#define ROUND_UP(x) (((x)+sizeof(long)-1) & ~(sizeof(long)-1))

static int stegfs_readdir(struct file * filp,
			 void * dirent, filldir_t filldir)
{
	int error = 0;
	unsigned long offset, blk;
	int i, num, stored;
	struct buffer_head * bh, * tmp, * bha[16];
	unsigned short iv, iv_tmp, iva[16];
	struct ext2_dir_entry_2 * de;
	struct super_block * sb;
	int err;
	struct inode *inode = filp->f_dentry->d_inode;
	char *bufplain;
	int isstegfs = 0;

	sb = inode->i_sb;

	stored = 0;
	bh = NULL;
	offset = filp->f_pos & (sb->s_blocksize - 1);

	isstegfs = inode->i_sb->u.stegfs_sb.s_x->s_isstegfs;

	if (STEGFS_IS_HID_INO(inode->i_ino)) {
		if (!inode->i_sb->u.stegfs_sb.s_x->s_slkeys[STEGFS_INO_LVL(inode->i_ino)-1])
			return 0;
	}

	if (inode->i_ino != STEGFS_HID_ROOT_INO) {
		while (!error && !stored && filp->f_pos < inode->i_size) {
			blk = (filp->f_pos) >> EXT2_BLOCK_SIZE_BITS(sb);
			bh = stegfs_bread (inode, blk, &iv, 0, &err);
			if (!bh) {
				stegfs_error (sb, "stegfs_readdir",
					      "directory #%lu contains a hole at offset %lu",
					      inode->i_ino, (unsigned long)filp->f_pos);
				filp->f_pos += sb->s_blocksize - offset;
				continue;
			}
			if (STEGFS_IS_HID_INO(inode->i_ino)) {
				bufplain = kmalloc(bh->b_size, GFP_KERNEL);
				stegfs_decrypt_cbc2(inode->i_sb,
						    STEGFS_INO_LVL(inode->i_ino),
						    bh->b_blocknr,
						    bh->b_data, bufplain, bh->b_size, iv);
			}
			else
				bufplain = bh->b_data;
				

			/*
			 * Do the readahead
			 */
			if (!offset) {
				for (i = 16 >> (EXT2_BLOCK_SIZE_BITS(sb) - 9), num = 0;
				     i > 0; i--) {
					tmp = stegfs_getblk (inode, ++blk,
							     &iv_tmp, 0, &err);
					if (tmp && !buffer_uptodate(tmp) && !buffer_locked(tmp)) {
						iva[num] = iv_tmp;
						bha[num++] = tmp;
					}
					else
						brelse (tmp);
				}
				if (num) {
					ll_rw_block (READA, num, bha);
					for (i = 0; i < num; i++)
						brelse (bha[i]);
				}
			}
		
revalidate:
			/* If the dir block has changed since the last call to
			 * readdir(2), then we might be pointing to an invalid
			 * dirent right now.  Scan from the start of the block
			 * to make sure. */
			if (filp->f_version != inode->i_version) {
				for (i = 0; i < sb->s_blocksize && i < offset; ) {
					de = (struct ext2_dir_entry_2 *) 
						(bufplain + i);
				/* It's too expensive to do a full
				 * dirent test each time round this
				 * loop, but we do have to test at
				 * least that it is non-zero.  A
				 * failure will be detected in the
				 * dirent test below. */
					if (le16_to_cpu(de->rec_len) < EXT2_DIR_REC_LEN(1))
						break;
					i += le16_to_cpu(de->rec_len);
				}
				offset = i;
				filp->f_pos = (filp->f_pos & ~(sb->s_blocksize - 1))
					| offset;
				filp->f_version = inode->i_version;
			}
		
			while (!error && filp->f_pos < inode->i_size 
			       && offset < sb->s_blocksize) {
				de = (struct ext2_dir_entry_2 *) (bufplain + offset);
				if (!stegfs_check_dir_entry ("stegfs_readdir", inode, de,
							     bufplain, offset)) {
				/* On error, skip the f_pos to the
                                   next block. */
stegfs_debug("error\n");
					filp->f_pos = (filp->f_pos & (sb->s_blocksize - 1))
						+ 1;
					brelse (bh);
					return stored;
				}
				offset += le16_to_cpu(de->rec_len);
				if (le32_to_cpu(de->inode)) {
				/* We might block in the next section
				 * if the data destination is
				 * currently swapped out.  So, use a
				 * version stamp to detect whether or
				 * not the directory has been modified
				 * during the copy operation.
				 * AV: It can't be modified, but it fscking
				 * can be seeked by another process that shares
				 * the descriptor.
 				 */
					unsigned long version = filp->f_version;

					error = filldir(dirent, de->name,
							de->name_len,
							filp->f_pos,
							le32_to_cpu(de->inode));
					if (error)
						break;
					if (version != filp->f_version)
						goto revalidate;
					stored ++;
				}

                                if(inode->i_ino == EXT2_ROOT_INO &&
				   (filp->f_pos + le16_to_cpu(de->rec_len))
				   == inode->i_sb->s_blocksize
				   && isstegfs) {

                                        /* add hidden root dir */
                                        error = filldir(dirent, "stegfs",
                                                        6, filp->f_pos + 
                                                        ROUND_UP(8+de->name_len+1),
                                                        STEGFS_HID_ROOT_INO);
                                        if(error) {
                                                stegfs_debug("error with EHRI\n");
						break;
					}

                                }

				filp->f_pos += le16_to_cpu(de->rec_len);
			}
			offset = 0;
			brelse (bh);
			if (STEGFS_IS_HID_INO(inode->i_ino))
				kfree(bufplain);
		}
	}
	else {
                if(inode->i_ino == STEGFS_HID_ROOT_INO &&
                   filp->f_pos == 0) {
                        char d_name[5];
                        stegfs_debug("hidden ino: %lu fpos: %lu\n",
				     inode->i_ino, filp->f_pos);
                        stored = 0;
                        
                        error = filldir(dirent, ".", 1, filp->f_pos,
                                        STEGFS_HID_ROOT_INO);
                        filp->f_pos += ROUND_UP(8+1+1);
                        stored++;

                        error += filldir(dirent, "..", 2, filp->f_pos,
                                         EXT2_ROOT_INO);
                        filp->f_pos += ROUND_UP(8+2+1);
                        stored++;

			for(i=0; i < (STEGFS_MAX_LEVELS-1); i++) {
				if (STEGFS_SB(sb)->s_x->s_slkeys[i]) {
					sprintf(d_name, "%u", i+1);

					error += filldir(dirent, d_name,
							 (i < 9)?1:2,
							 filp->f_pos,
							 EXT2_ROOT_INO|STEGFS_INO_MASK|
							 ((stored-1)<<STEGFS_LVL_SHIFT));
					filp->f_pos += ROUND_UP(8+((i < 9)?1:2)+1);
					stored++;
				}
/*
                                error += filldir(dirent, d_name,
						 (stored-1 < 10)?1:2,
                                                 filp->f_pos,
						 EXT2_ROOT_INO);
                                filp->f_pos += ROUND_UP(8+((stored-1 < 10)?1:2)
							+1);
*/                              
                        }

                        filp->f_pos = inode->i_sb->s_blocksize;
                }


	}
	UPDATE_ATIME(inode);
	return 0;
}
