/*
 *  linux/fs/stegfs/balloc.c
 *
 * Copyright (C) 1999-2001
 * Andrew McDonald (andrew@mcdonald.org.uk)
 *
 *  from
 *
 *  linux/fs/ext2/balloc.c
 *
 * Copyright (C) 1992, 1993, 1994, 1995
 * Remy Card (card@masi.ibp.fr)
 * Laboratoire MASI - Institut Blaise Pascal
 * Universite Pierre et Marie Curie (Paris VI)
 *
 *  Enhanced block allocation by Stephen Tweedie (sct@dcs.ed.ac.uk), 1993
 *  Big-endian to little-endian byte-swapping/bitmaps by
 *        David S. Miller (davem@caip.rutgers.edu), 1995
 */

/*
 * balloc.c contains the blocks allocation and deallocation routines
 */

/*
 * The free blocks are managed by bitmaps.  A file system contains several
 * blocks groups.  Each group contains 1 bitmap block for blocks, 1 bitmap
 * block for inodes, N blocks for the inode table and data blocks.
 *
 * The file system contains group descriptors which are located after the
 * super block.  Each descriptor contains the number of the bitmap block and
 * the free blocks count in the block.  The descriptors are loaded in memory
 * when a file system is mounted (see stegfs_read_super).
 */

#include "fs.h"
#include "stegfs_fs.h"
#include <linux/stat.h>
#include <linux/sched.h>
#include <linux/string.h>
#include <linux/locks.h>
#include <linux/quotaops.h>
#include <linux/random.h>
#include <linux/malloc.h>

#include <asm/bitops.h>
#include <asm/byteorder.h>
#include <asm/uaccess.h>

#if defined(CONFIG_CIPHERS) || defined(CONFIG_CIPHERS_MODULE)
#include <linux/crypto.h>
#endif

#define in_range(b, first, len)		((b) >= (first) && (b) <= (first) + (len) - 1)

int stegfs_blkused(struct super_block *, int);
int stegfs_blkused_ext2(struct super_block *sb, int bnum);

struct ext2_group_desc * stegfs_get_group_desc(struct super_block * sb,
					     unsigned int block_group,
					     struct buffer_head ** bh)
{
	unsigned long group_desc;
	unsigned long desc;
	struct ext2_group_desc * gdp;

	if (block_group >= sb->u.stegfs_sb.s_groups_count) {
		stegfs_error (sb, "stegfs_get_group_desc",
			    "block_group >= groups_count - "
			    "block_group = %d, groups_count = %lu",
			    block_group, sb->u.stegfs_sb.s_groups_count);

		return NULL;
	}
	
	group_desc = block_group / STEGFS_DESC_PER_BLOCK(sb);
	desc = block_group % STEGFS_DESC_PER_BLOCK(sb);
	if (!sb->u.stegfs_sb.s_group_desc[group_desc]) {
		stegfs_error (sb, "stegfs_get_group_desc",
			    "Group descriptor not loaded - "
			    "block_group = %d, group_desc = %lu, desc = %lu",
			     block_group, group_desc, desc);
		return NULL;
	}
	
	gdp = (struct ext2_group_desc *) 
	      sb->u.stegfs_sb.s_group_desc[group_desc]->b_data;
	if (bh)
		*bh = sb->u.stegfs_sb.s_group_desc[group_desc];
	return gdp + desc;
}

/*
 * Read the bitmap for a given block_group, reading into the specified 
 * slot in the superblock's bitmap cache.
 *
 * Return >=0 on success or a -ve error code.
 */

static int read_block_bitmap (struct super_block * sb,
			       unsigned int block_group,
			       unsigned long bitmap_nr)
{
	struct ext2_group_desc * gdp;
	struct buffer_head * bh = NULL;
	int retval = -EIO;
	
	gdp = stegfs_get_group_desc (sb, block_group, NULL);
	if (!gdp)
		goto error_out;
	retval = 0;
	bh = bread (sb->s_dev, le32_to_cpu(gdp->bg_block_bitmap), sb->s_blocksize);
	if (!bh) {
		stegfs_error (sb, "read_block_bitmap",
			    "Cannot read block bitmap - "
			    "block_group = %d, block_bitmap = %lu",
			    block_group, (unsigned long) gdp->bg_block_bitmap);
		retval = -EIO;
	}
	/*
	 * On IO error, just leave a zero in the superblock's block pointer for
	 * this group.  The IO will be retried next time.
	 */
error_out:
	sb->u.stegfs_sb.s_block_bitmap_number[bitmap_nr] = block_group;
	sb->u.stegfs_sb.s_block_bitmap[bitmap_nr] = bh;
	return retval;
}

/*
 * load_block_bitmap loads the block bitmap for a blocks group
 *
 * It maintains a cache for the last bitmaps loaded.  This cache is managed
 * with a LRU algorithm.
 *
 * Notes:
 * 1/ There is one cache per mounted file system.
 * 2/ If the file system contains less than EXT2_MAX_GROUP_LOADED groups,
 *    this function reads the bitmap without maintaining a LRU cache.
 * 
 * Return the slot used to store the bitmap, or a -ve error code.
 */
static int load__block_bitmap (struct super_block * sb,
			       unsigned int block_group)
{
	int i, j, retval = 0;
	unsigned long block_bitmap_number;
	struct buffer_head * block_bitmap;

	if (block_group >= sb->u.stegfs_sb.s_groups_count)
		stegfs_panic (sb, "load_block_bitmap",
			    "block_group >= groups_count - "
			    "block_group = %d, groups_count = %lu",
			    block_group, sb->u.stegfs_sb.s_groups_count);

	if (sb->u.stegfs_sb.s_groups_count <= EXT2_MAX_GROUP_LOADED) {
		if (sb->u.stegfs_sb.s_block_bitmap[block_group]) {
			if (sb->u.stegfs_sb.s_block_bitmap_number[block_group] ==
			    block_group)
				return block_group;
			stegfs_error (sb, "load_block_bitmap",
				    "block_group != block_bitmap_number");
		}
		retval = read_block_bitmap (sb, block_group, block_group);
		if (retval < 0)
			return retval;
		return block_group;
	}

	for (i = 0; i < sb->u.stegfs_sb.s_loaded_block_bitmaps &&
		    sb->u.stegfs_sb.s_block_bitmap_number[i] != block_group; i++)
		;
	if (i < sb->u.stegfs_sb.s_loaded_block_bitmaps &&
  	    sb->u.stegfs_sb.s_block_bitmap_number[i] == block_group) {
		block_bitmap_number = sb->u.stegfs_sb.s_block_bitmap_number[i];
		block_bitmap = sb->u.stegfs_sb.s_block_bitmap[i];
		for (j = i; j > 0; j--) {
			sb->u.stegfs_sb.s_block_bitmap_number[j] =
				sb->u.stegfs_sb.s_block_bitmap_number[j - 1];
			sb->u.stegfs_sb.s_block_bitmap[j] =
				sb->u.stegfs_sb.s_block_bitmap[j - 1];
		}
		sb->u.stegfs_sb.s_block_bitmap_number[0] = block_bitmap_number;
		sb->u.stegfs_sb.s_block_bitmap[0] = block_bitmap;

		/*
		 * There's still one special case here --- if block_bitmap == 0
		 * then our last attempt to read the bitmap failed and we have
		 * just ended up caching that failure.  Try again to read it.
		 */
		if (!block_bitmap)
			retval = read_block_bitmap (sb, block_group, 0);
	} else {
		if (sb->u.stegfs_sb.s_loaded_block_bitmaps < EXT2_MAX_GROUP_LOADED)
			sb->u.stegfs_sb.s_loaded_block_bitmaps++;
		else
			brelse (sb->u.stegfs_sb.s_block_bitmap[EXT2_MAX_GROUP_LOADED - 1]);
		for (j = sb->u.stegfs_sb.s_loaded_block_bitmaps - 1; j > 0;  j--) {
			sb->u.stegfs_sb.s_block_bitmap_number[j] =
				sb->u.stegfs_sb.s_block_bitmap_number[j - 1];
			sb->u.stegfs_sb.s_block_bitmap[j] =
				sb->u.stegfs_sb.s_block_bitmap[j - 1];
		}
		retval = read_block_bitmap (sb, block_group, 0);
	}
	return retval;
}

/*
 * Load the block bitmap for a given block group.  First of all do a couple
 * of fast lookups for common cases and then pass the request onto the guts
 * of the bitmap loader.
 *
 * Return the slot number of the group in the superblock bitmap cache's on
 * success, or a -ve error code.
 *
 * There is still one inconsistancy here --- if the number of groups in this
 * filesystems is <= EXT2_MAX_GROUP_LOADED, then we have no way of 
 * differentiating between a group for which we have never performed a bitmap
 * IO request, and a group for which the last bitmap read request failed.
 */
static inline int load_block_bitmap (struct super_block * sb,
				     unsigned int block_group)
{
	int slot;
	
	/*
	 * Do the lookup for the slot.  First of all, check if we're asking
	 * for the same slot as last time, and did we succeed that last time?
	 */
	if (sb->u.stegfs_sb.s_loaded_block_bitmaps > 0 &&
	    sb->u.stegfs_sb.s_block_bitmap_number[0] == block_group &&
	    sb->u.stegfs_sb.s_block_bitmap[0]) {
		return 0;
	}
	/*
	 * Or can we do a fast lookup based on a loaded group on a filesystem
	 * small enough to be mapped directly into the superblock?
	 */
	else if (sb->u.stegfs_sb.s_groups_count <= EXT2_MAX_GROUP_LOADED && 
		 sb->u.stegfs_sb.s_block_bitmap_number[block_group] == block_group &&
		 sb->u.stegfs_sb.s_block_bitmap[block_group]) {
		slot = block_group;
	}
	/*
	 * If not, then do a full lookup for this block group.
	 */
	else {
		slot = load__block_bitmap (sb, block_group);
	}

	/*
	 * <0 means we just got an error
	 */
	if (slot < 0)
		return slot;
	
	/*
	 * If it's a valid slot, we may still have cached a previous IO error,
	 * in which case the bh in the superblock cache will be zero.
	 */
	if (!sb->u.stegfs_sb.s_block_bitmap[slot])
		return -EIO;
	
	/*
	 * Must have been read in OK to get this far.
	 */
	return slot;
}

void stegfs_free_blocks1 (const struct inode * inode, unsigned long block,
		       unsigned long count)
{
	struct buffer_head * bh;
	struct buffer_head * bh2;
	unsigned long block_group;
	unsigned long bit;
	unsigned long i;
	int bitmap_nr;
	unsigned long overflow;
	struct super_block * sb;
	struct ext2_group_desc * gdp;
	struct ext2_super_block * es;
	struct stegfs_btable btab;

	stegfs_debug("ino: %lu blk: %lu count: %lu\n", inode->i_ino,
		     block, count);

	sb = inode->i_sb;
	if (!sb) {
		printk ("stegfs_free_blocks: nonexistent device");
		return;
	}
	lock_super (sb);
	es = sb->u.stegfs_sb.s_es;
	if (block < le32_to_cpu(es->s_first_data_block) || 
	    (block + count) > le32_to_cpu(es->s_blocks_count)) {
		stegfs_error (sb, "stegfs_free_blocks",
			    "Freeing blocks not in datazone - "
			    "block = %lu, count = %lu", block, count);
		goto error_return;
	}

	stegfs_debug ("freeing block %lu\n", block);

	/* secure delete the contents of the blocks */
	for (i = block; i < block+count; i++) {
		struct buffer_head * bh0 =
			getblk (inode->i_dev, i,
				inode->i_sb->s_blocksize);

#ifdef STEGFS_DEL_REAL_RAND
		get_random_bytes(bh0->b_data, bh0->b_size);
#else
#ifdef STEGFS_DEL_ZERO
		memset(bh0->b_data, 0, bh0->b_size);
#else

#if defined(CONFIG_CIPHERS) || defined(CONFIG_CIPHERS_MODULE)
		{
                        struct cipher_implementation *ci;
                        struct cipher_context *cx;
                        char key[STEGFS_KEY_BYTES];
                        char *tmpbuf;

                        get_random_bytes(key, 16);
                        tmpbuf = (char *) kmalloc(bh0->b_size, GFP_KERNEL);
                        /* don't actually care what tmpbuf contains */
                        cx = (struct cipher_context *)
                                kmalloc(sizeof(struct cipher_context),
                                        GFP_KERNEL);
                        memset(cx, 0, sizeof(struct cipher_context));
                        ci = STEGFS_SB(sb)->s_x->s_ci;
                        cx->ci = ci;
                        cx->keyinfo = (u32 *) kmalloc(cx->ci->key_schedule_size,
                                                      GFP_KERNEL);
                        ci->set_key(cx, key, STEGFS_KEY_BYTES);
                        ci->encrypt(cx, tmpbuf,
                                    bh0->b_data, bh0->b_size);
                        memset(cx->keyinfo, 0, cx->ci->key_schedule_size);
                        kfree(cx->keyinfo);
                        memset(cx, 0, sizeof(*cx));
                        kfree(cx);
                        memset(key, 0, STEGFS_KEY_BYTES);
                        memset(tmpbuf, 0, bh0->b_size);
                        kfree(tmpbuf);
		}
#else
		{
			char *keys;
			char key[16];
			char *p;
			char inb[16], outb[16];
			int si;

			get_random_bytes(key, 16);
			keys = STEGFS_SB(sb)->s_x->s_cipher->set_key(key, 128);
			for(p=bh0->b_data; p<(bh0->b_data+bh0->b_size); p+=16) {
				if (p>bh0->b_data) {
					for(si=0; si<16; si++)
						inb[si] = p[si]^outb[si];
				}
				else
					memcpy(inb, p, 16);
				STEGFS_SB(sb)->s_x->s_cipher->encrypt(keys, inb, outb);
				memcpy(p, outb, 16);
			}
			kfree(keys);
		}
#endif
#endif
#endif
		mark_buffer_dirty(bh0,1);
		/* force write? */

		if (STEGFS_SB(sb)->s_x->s_isstegfs) {
			get_random_bytes(&btab, sizeof(struct stegfs_btable));
			stegfs_put_btab(sb, bh0->b_blocknr, &btab);
		}

		brelse(bh0);
	}

	if (!STEGFS_IS_HID_INO(inode->i_ino)) {
do_more:
		overflow = 0;
		block_group = (block - le32_to_cpu(es->s_first_data_block)) /
			STEGFS_BLOCKS_PER_GROUP(sb);
		bit = (block - le32_to_cpu(es->s_first_data_block)) %
			STEGFS_BLOCKS_PER_GROUP(sb);
		/*
		 * Check to see if we are freeing blocks across a group
		 * boundary.
		 */
		if (bit + count > STEGFS_BLOCKS_PER_GROUP(sb)) {
			overflow = bit + count - STEGFS_BLOCKS_PER_GROUP(sb);
			count -= overflow;
		}
		bitmap_nr = load_block_bitmap (sb, block_group);
		if (bitmap_nr < 0)
			goto error_return;
	
		bh = sb->u.stegfs_sb.s_block_bitmap[bitmap_nr];
		gdp = stegfs_get_group_desc (sb, block_group, &bh2);
		if (!gdp)
			goto error_return;

		if (test_opt (sb, CHECK_STRICT) &&
		    (in_range (le32_to_cpu(gdp->bg_block_bitmap), block, count) ||
		     in_range (le32_to_cpu(gdp->bg_inode_bitmap), block, count) ||
		     in_range (block, le32_to_cpu(gdp->bg_inode_table),
			       sb->u.stegfs_sb.s_itb_per_group) ||
		     in_range (block + count - 1, le32_to_cpu(gdp->bg_inode_table),
			       sb->u.stegfs_sb.s_itb_per_group)))
			stegfs_panic (sb, "stegfs_free_blocks",
				      "Freeing blocks in system zones - "
				      "Block = %lu, count = %lu",
				      block, count);

		for (i = 0; i < count; i++) {
			if (!ext2_clear_bit (bit + i, bh->b_data))
				stegfs_warning (sb, "stegfs_free_blocks",
						"bit already cleared for block %lu", 
						block);
			else {
				DQUOT_FREE_BLOCK(sb, inode, 1);
				gdp->bg_free_blocks_count =
					cpu_to_le16(le16_to_cpu(gdp->bg_free_blocks_count)+1);
				es->s_free_blocks_count =
					cpu_to_le32(le32_to_cpu(es->s_free_blocks_count)+1);
			}
		}
	
		mark_buffer_dirty(bh2, 1);
		mark_buffer_dirty(sb->u.stegfs_sb.s_x->s_sbh, 1);

		mark_buffer_dirty(bh, 1);
		if (sb->s_flags & MS_SYNCHRONOUS) {
			ll_rw_block (WRITE, 1, &bh);
			wait_on_buffer (bh);
		}
		if (overflow) {
			block += count;
			count = overflow;
			goto do_more;
		}
	}

	sb->s_dirt = 1;
error_return:
	unlock_super (sb);
	return;
}

void stegfs_free_blocks (const struct inode * inode, unsigned long block,
			 unsigned long count)
{
	struct stegfs_btable btab, btab2;
	int i, j, k;
	struct super_block * sb;

	if (!STEGFS_IS_HID_INO(inode->i_ino)) {
		stegfs_free_blocks1(inode, block, count);
		return;
	}

	if (!block && count == 1)
		return;

	sb = inode->i_sb;

	for(i=block; i<block+count; i++) {
		stegfs_get_btab(inode->i_sb, i, &btab);
		stegfs_decrypt_btab(inode->i_sb,
				    STEGFS_INO_LVL(inode->i_ino),
				    i, &btab);

		if(btab.magic1 == 0 &&
		   (btab.magic2 == 0 || btab.magic2 == 1) &&
		   btab.ino == inode->i_ino && !stegfs_blkused_ext2(sb, i)) {
			/* doesn't matter if used and freed by ext2 since
			   will just be overwriting unused block */
			stegfs_free_blocks1(inode, i, 1);
		}
		else {
			/* can we overwrite btab? */
			stegfs_get_btab(inode->i_sb, i, &btab);
			k = 0;
			for (j=0; j<(STEGFS_MAX_LEVELS-1); j++) {
				if (STEGFS_SB(inode->i_sb)->s_x->s_slkeys[j]) {
					stegfs_decrypt_btab2(inode->i_sb, j+1, i,
							     &btab, &btab2);
					if (btab2.magic1 == 0 &&
					    (btab2.magic2 == 0 || btab2.magic2 == 1) &&
					    btab2.ino != inode->i_ino) {
						k = 1;
						break;
					}
				}
			}
			if (!k) {
				get_random_bytes(&btab,
						 sizeof(struct stegfs_btable));
				stegfs_put_btab(sb, i, &btab);
			}
		}
	}

	return;
}

/*
 * stegfs_new_block uses a goal block to assist allocation.  If the goal is
 * free, or there is a free block within 32 blocks of the goal, that block
 * is allocated.  Otherwise a forward search is made for a free block; within 
 * each block group the search first looks for an entire free byte in the block
 * bitmap, and then for any free bit if that fails.
 */
int stegfs_new_block (const struct inode * inode, unsigned long goal,
		    u32 * prealloc_count, u32 * prealloc_block, int * err)
{
	struct buffer_head * bh;
	struct buffer_head * bh2;
	char * p, * r;
	int i, j, k, tmp;
	int bitmap_nr;
	struct super_block * sb;
	struct ext2_group_desc * gdp;
	struct ext2_super_block * es;
	struct stegfs_btable btab;
	int isstegfs = 0;

#ifdef STEGFS_DEBUG
        static int goal_hits = 0, goal_attempts = 0;
#endif

	*err = -ENOSPC;

	sb = inode->i_sb;
	if (!sb) {
		printk ("stegfs_new_block: nonexistent device");
		return 0;
	}

	lock_super (sb);
	es = sb->u.stegfs_sb.s_es;
#if 0
#ifdef STEGFS_DEBUG
	if(!STEGFS_IS_HID_INO(inode->i_ino))
		stegfs_debug("ino: %lu goal: %lu pcount: %u pblock: %u\n",
		     inode->i_ino, goal, *prealloc_count, *prealloc_block);
#endif
#endif
	if (STEGFS_IS_HID_INO(inode->i_ino)) {
#ifdef STEGFS_OLD_BALLOC
		j = stegfs_rand_new_block(sb);
		if (!(bh = getblk (sb->s_dev, j, sb->s_blocksize))) {
			stegfs_error (sb, "stegfs_new_block", "cannot get block %d", j);
			unlock_super (sb);
			return 0;
		}
		if (!buffer_uptodate(bh))
			wait_on_buffer(bh);
		memset(bh->b_data, 0, sb->s_blocksize);
		btab.magic1 = 0;
		btab.magic2 = 0;
		btab.iv = -1;
		btab.ino = inode->i_ino;
		stegfs_encrypt_cbc(inode->i_sb,
				   STEGFS_INO_LVL(inode->i_ino),
				   j,
				   bh->b_data, bh->b_size,
				   &btab.iv);
		btab.bchecksum = stegfs_chksum(bh->b_data,
					       bh->b_size);
		stegfs_encrypt_btab(inode->i_sb,
				    STEGFS_INO_LVL(inode->i_ino),
				    j, &btab);
		stegfs_put_btab(inode->i_sb, j, &btab);
		mark_buffer_uptodate(bh, 1);
		mark_buffer_dirty(bh, 1);
		brelse (bh);

		sb->s_dirt = 1;
		unlock_super (sb);
		*err = 0;
		return j;
#else

		stegfs_debug("block goal: %lu\n", goal);
		if (goal < 1 || goal > STEGFS_SB(sb)->s_es->s_blocks_count - 1
		    || stegfs_blkused(sb,goal)) {
			j = stegfs_rand_new_block(sb);
		}
		else {
			j = goal;
		}


		if (!(bh = getblk (sb->s_dev, j, sb->s_blocksize))) {
			stegfs_error (sb, "stegfs_new_block", "cannot get block %d", j);
			unlock_super (sb);
			return 0;
		}
		if (!buffer_uptodate(bh))
			wait_on_buffer(bh);
		memset(bh->b_data, 0, sb->s_blocksize);
		btab.magic1 = 0;
		btab.magic2 = 0;
		btab.iv = -1;
		btab.ino = inode->i_ino;
		stegfs_encrypt_cbc(inode->i_sb,
				   STEGFS_INO_LVL(inode->i_ino),
				   j,
				   bh->b_data, bh->b_size,
				   &btab.iv);
		btab.bchecksum = stegfs_chksum(bh->b_data,
					       bh->b_size);
		stegfs_encrypt_btab(inode->i_sb,
				    STEGFS_INO_LVL(inode->i_ino),
				    j, &btab);
		stegfs_put_btab(inode->i_sb, j, &btab);
		mark_buffer_uptodate(bh, 1);
		mark_buffer_dirty(bh, 1);
		brelse (bh);

		sb->s_dirt = 1;
		unlock_super (sb);
		*err = 0;
		return j;

#endif
	}

	if (le32_to_cpu(es->s_free_blocks_count) <= le32_to_cpu(es->s_r_blocks_count) &&
	    ((sb->u.stegfs_sb.s_resuid != current->fsuid) &&
	     (sb->u.stegfs_sb.s_resgid == 0 ||
	      !in_group_p (sb->u.stegfs_sb.s_resgid)) && 
	     !capable(CAP_SYS_RESOURCE))) {
		unlock_super (sb);
		return 0;
	}

	stegfs_debug ("goal=%lu.\n", goal);

	isstegfs = STEGFS_SB(sb)->s_x->s_isstegfs;

repeat:
	/*
	 * First, test whether the goal block is free.
	 */
	if (goal < le32_to_cpu(es->s_first_data_block) ||
	    goal >= le32_to_cpu(es->s_blocks_count))
		goal = le32_to_cpu(es->s_first_data_block);
	i = (goal - le32_to_cpu(es->s_first_data_block)) / STEGFS_BLOCKS_PER_GROUP(sb);
	gdp = stegfs_get_group_desc (sb, i, &bh2);
	if (!gdp)
		goto io_error;

	if (le16_to_cpu(gdp->bg_free_blocks_count) > 0) {
		j = ((goal - le32_to_cpu(es->s_first_data_block)) % STEGFS_BLOCKS_PER_GROUP(sb));
#ifdef STEGFS_DEBUG
		if (j)
			goal_attempts++;
#endif
		bitmap_nr = load_block_bitmap (sb, i);
		if (bitmap_nr < 0)
			goto io_error;
		
		bh = sb->u.stegfs_sb.s_block_bitmap[bitmap_nr];

		stegfs_debug ("goal is at %d:%d.\n", i, j);

		if ((!isstegfs && !ext2_test_bit(j, bh->b_data)) ||
		    (isstegfs && !stegfs_blkused(sb, goal))) {
#ifdef STEGFS_DEBUG
			goal_hits++;
			stegfs_debug ("goal bit allocated.\n");
#endif
			goto got_block;
		}
		if (j) {
			/*
			 * The goal was occupied; search forward for a free 
			 * block within the next XX blocks.
			 *
			 * end_goal is more or less random, but it has to be
			 * less than STEGFS_BLOCKS_PER_GROUP. Aligning up to the
			 * next 64-bit boundary is simple..
			 */
			int end_goal = (j + 63) & ~63;
			j = ext2_find_next_zero_bit(bh->b_data, end_goal, j);
			while (isstegfs && j < end_goal &&
			       stegfs_blkused(sb, j + i * STEGFS_BLOCKS_PER_GROUP(sb)
					      + le32_to_cpu(es->s_first_data_block))) {
				j++;
				j = ext2_find_next_zero_bit(bh->b_data, end_goal, j);
			}

			if (j < end_goal)
				goto got_block;
		}
	
		stegfs_debug ("Bit not found near goal\n");

		/*
		 * There has been no free block found in the near vicinity
		 * of the goal: do a search forward through the block groups,
		 * searching in each group first for an entire free byte in
		 * the bitmap and then for any free bit.
		 * 
		 * Search first in the remainder of the current group; then,
		 * cyclicly search through the rest of the groups.
		 */
		p = ((char *) bh->b_data) + (j >> 3);
		r = memscan(p, 0, (STEGFS_BLOCKS_PER_GROUP(sb) - j + 7) >> 3);
		k = (r - ((char *) bh->b_data)) << 3;
		while (isstegfs && k < STEGFS_BLOCKS_PER_GROUP(sb) &&
		       stegfs_blkused(sb, k + i * STEGFS_BLOCKS_PER_GROUP(sb)
				      + le32_to_cpu(es->s_first_data_block))) {
			r = memscan(++r, 0, (STEGFS_BLOCKS_PER_GROUP(sb) - k + 7) >> 3);
			k = (r - ((char *) bh->b_data)) << 3;
		}
		if (k < STEGFS_BLOCKS_PER_GROUP(sb)) {
			j = k;
			goto search_back;
		}

		k = ext2_find_next_zero_bit ((unsigned long *) bh->b_data, 
					STEGFS_BLOCKS_PER_GROUP(sb),
					j);
		while (isstegfs && k < STEGFS_BLOCKS_PER_GROUP(sb) &&
		       stegfs_blkused(sb, k + i * STEGFS_BLOCKS_PER_GROUP(sb)
				      + le32_to_cpu(es->s_first_data_block))) {
			k++;
			k = ext2_find_next_zero_bit ((unsigned long *) bh->b_data, 
						     STEGFS_BLOCKS_PER_GROUP(sb),k);
		}
		if (k < STEGFS_BLOCKS_PER_GROUP(sb)) {
			j = k;
			goto got_block;
		}
	}

	stegfs_debug ("Bit not found in block group %d.\n", i);

	/*
	 * Now search the rest of the groups.  We assume that 
	 * i and gdp correctly point to the last group visited.
	 */
try_next_grp:
	for (k = 0; k < sb->u.stegfs_sb.s_groups_count; k++) {
		i++;
		if (i >= sb->u.stegfs_sb.s_groups_count)
			i = 0;
		gdp = stegfs_get_group_desc (sb, i, &bh2);
		if (!gdp) {
			*err = -EIO;
			unlock_super (sb);
			return 0;
		}
		if (le16_to_cpu(gdp->bg_free_blocks_count) > 0)
			break;
	}
	if (k >= sb->u.stegfs_sb.s_groups_count) {
		unlock_super (sb);
		return 0;
	}
	bitmap_nr = load_block_bitmap (sb, i);
	if (bitmap_nr < 0)
		goto io_error;
	
	bh = sb->u.stegfs_sb.s_block_bitmap[bitmap_nr];
	r = memscan(bh->b_data, 0, STEGFS_BLOCKS_PER_GROUP(sb) >> 3);
	j = (r - bh->b_data) << 3;
	while (isstegfs && j < STEGFS_BLOCKS_PER_GROUP(sb) &&
	       stegfs_blkused(sb, j + i * STEGFS_BLOCKS_PER_GROUP(sb)
			      + le32_to_cpu(es->s_first_data_block))) {
		++r;
		r = memscan(r, 0, (STEGFS_BLOCKS_PER_GROUP(sb) >> 3) -
			            (r - bh->b_data));
		j = (r - bh->b_data) << 3;
	}
	if (j < STEGFS_BLOCKS_PER_GROUP(sb))
		goto search_back;
	else {
		j = ext2_find_first_zero_bit ((unsigned long *) bh->b_data,
					      STEGFS_BLOCKS_PER_GROUP(sb));
		while (isstegfs && j < STEGFS_BLOCKS_PER_GROUP(sb) &&
		       stegfs_blkused(sb, j + i * STEGFS_BLOCKS_PER_GROUP(sb)
				      + le32_to_cpu(es->s_first_data_block))) {
			++j;
			j = ext2_find_next_zero_bit ((unsigned long *) bh->b_data,
						     STEGFS_BLOCKS_PER_GROUP(sb),j);
		}
	}
	if (j >= STEGFS_BLOCKS_PER_GROUP(sb)) {
		if (!isstegfs) {
			stegfs_error (sb, "stegfs_new_block",
			    "Free blocks count corrupted for block group %d", i);
			unlock_super (sb);
			return 0;
		}
		goto try_next_grp;
	}

search_back:
	/* 
	 * We have succeeded in finding a free byte in the block
	 * bitmap.  Now search backwards up to 7 bits to find the
	 * start of this group of free blocks.
	 */

	if (!isstegfs)
		for (k = 0; k < 7 && j > 0 && !ext2_test_bit (j - 1, bh->b_data); k++, j--);
	else {
		for (k = 0; k < 7 && j > 0 &&
			     !stegfs_blkused(sb, j-1 +
					     i * STEGFS_BLOCKS_PER_GROUP(sb) +
					     le32_to_cpu(es->s_first_data_block));
					     k++, j--);
	}

got_block:

	stegfs_debug ("using block group %d(%d)\n", i, gdp->bg_free_blocks_count);

	/*
	 * Check quota for allocation of this block.
	 */
	if(DQUOT_ALLOC_BLOCK(sb, inode, 1)) {
		unlock_super(sb);
		*err = -EDQUOT;
		return 0;
	}

	tmp = j + i * STEGFS_BLOCKS_PER_GROUP(sb) + le32_to_cpu(es->s_first_data_block);

	if (test_opt (sb, CHECK_STRICT) &&
	    (tmp == le32_to_cpu(gdp->bg_block_bitmap) ||
	     tmp == le32_to_cpu(gdp->bg_inode_bitmap) ||
	     in_range (tmp, le32_to_cpu(gdp->bg_inode_table), sb->u.stegfs_sb.s_itb_per_group)))
		stegfs_panic (sb, "stegfs_new_block",
			    "Allocating block in system zone - "
			    "block = %u", tmp);

	if (ext2_set_bit (j, bh->b_data)) {
		stegfs_warning (sb, "stegfs_new_block",
			      "bit already set for block %d", j);
		DQUOT_FREE_BLOCK(sb, inode, 1);
		goto repeat;
	}

	stegfs_debug ("found bit %d\n", j);

	/*
	 * Do block preallocation now if required.
	 */
#ifdef EXT2_PREALLOCATE
	if (prealloc_count && !*prealloc_count) {
		int	prealloc_goal;
		unsigned long   next_block = tmp + 1;

		prealloc_goal = es->s_prealloc_blocks ?
			es->s_prealloc_blocks : EXT2_DEFAULT_PREALLOC_BLOCKS;

		*prealloc_block = next_block;
		for (k = 1;
		     k < prealloc_goal && (j + k) < STEGFS_BLOCKS_PER_GROUP(sb);
		     k++, next_block++) {
			if (DQUOT_PREALLOC_BLOCK(sb, inode, 1))
				break;
			if (*prealloc_block + *prealloc_count != next_block ||
			    ext2_set_bit (j + k, bh->b_data)) {
				DQUOT_FREE_BLOCK(sb, inode, 1);
 				break;
			}
			(*prealloc_count)++;
		}	
		gdp->bg_free_blocks_count =
			cpu_to_le16(le16_to_cpu(gdp->bg_free_blocks_count) -
			       (k - 1));
		es->s_free_blocks_count =
			cpu_to_le32(le32_to_cpu(es->s_free_blocks_count) -
			       (k - 1));
		stegfs_debug ("Preallocated a further %u bits.\n",
			      (k - 1));
	}
#endif

	j = tmp;

	mark_buffer_dirty(bh, 1);
	if (sb->s_flags & MS_SYNCHRONOUS) {
		ll_rw_block (WRITE, 1, &bh);
		wait_on_buffer (bh);
	}

	if (j >= le32_to_cpu(es->s_blocks_count)) {
		stegfs_error (sb, "stegfs_new_block",
			    "block >= blocks count - "
			    "block_group = %d, block=%d", i, j);
		unlock_super (sb);
		return 0;
	}
	if (!(bh = getblk (sb->s_dev, j, sb->s_blocksize))) {
		stegfs_error (sb, "stegfs_new_block", "cannot get block %d", j);
		unlock_super (sb);
		return 0;
	}
	if (!buffer_uptodate(bh))
		wait_on_buffer(bh);
	memset(bh->b_data, 0, sb->s_blocksize);
	/* FIXME: randomize btab entry */
	mark_buffer_uptodate(bh, 1);
	mark_buffer_dirty(bh, 1);
	brelse (bh);
#ifdef STEGFS_DEBUG
	stegfs_debug ("allocating block %d. "
		    "Goal hits %d of %d.\n", j, goal_hits, goal_attempts);
#endif
	gdp->bg_free_blocks_count = cpu_to_le16(le16_to_cpu(gdp->bg_free_blocks_count) - 1);
	mark_buffer_dirty(bh2, 1);
	es->s_free_blocks_count = cpu_to_le32(le32_to_cpu(es->s_free_blocks_count) - 1);
	mark_buffer_dirty(sb->u.stegfs_sb.s_x->s_sbh, 1);
	sb->s_dirt = 1;
	unlock_super (sb);
	*err = 0;
	return j;
	
io_error:
	*err = -EIO;
	unlock_super (sb);
	return 0;
	
}

unsigned long stegfs_count_free_blocks (struct super_block * sb)
{
#ifdef STEGFS_DEBUG
	struct ext2_super_block * es;
	unsigned long desc_count, bitmap_count, x;
	int bitmap_nr;
	struct ext2_group_desc * gdp;
	int i;
	
	lock_super (sb);
	es = sb->u.stegfs_sb.s_es;
	desc_count = 0;
	bitmap_count = 0;
	gdp = NULL;
	for (i = 0; i < sb->u.stegfs_sb.s_groups_count; i++) {
		gdp = stegfs_get_group_desc (sb, i, NULL);
		if (!gdp)
			continue;
		desc_count += le16_to_cpu(gdp->bg_free_blocks_count);
		bitmap_nr = load_block_bitmap (sb, i);
		if (bitmap_nr < 0)
			continue;
		
		x = stegfs_count_free (sb->u.stegfs_sb.s_block_bitmap[bitmap_nr],
				     sb->s_blocksize);
		printk ("group %d: stored = %d, counted = %lu\n",
			i, le16_to_cpu(gdp->bg_free_blocks_count), x);
		bitmap_count += x;
	}
	printk("stegfs_count_free_blocks: stored = %u, computed = %lu, %lu\n",
	       le32_to_cpu(es->s_free_blocks_count), desc_count, bitmap_count);
	unlock_super (sb);
	return bitmap_count;
#else
	return le32_to_cpu(sb->u.stegfs_sb.s_es->s_free_blocks_count);
#endif
}

static inline int block_in_use (unsigned long block,
				struct super_block * sb,
				unsigned char * map)
{
	return ext2_test_bit ((block - le32_to_cpu(sb->u.stegfs_sb.s_es->s_first_data_block)) %
			 STEGFS_BLOCKS_PER_GROUP(sb), map);
}

static int test_root(int a, int b)
{
	if (a == 0)
		return 1;
	while (1) {
		if (a == 1)
			return 1;
		if (a % b)
			return 0;
		a = a / b;
	}
}

int stegfs_group_sparse(int group)
{
	return (test_root(group, 3) || test_root(group, 5) ||
		test_root(group, 7));
}

void stegfs_check_blocks_bitmap (struct super_block * sb)
{
	struct buffer_head * bh;
	struct ext2_super_block * es;
	unsigned long desc_count, bitmap_count, x;
	unsigned long desc_blocks;
	int bitmap_nr;
	struct ext2_group_desc * gdp;
	int i, j;

	lock_super (sb);
	es = sb->u.stegfs_sb.s_es;
	desc_count = 0;
	bitmap_count = 0;
	gdp = NULL;
	desc_blocks = (sb->u.stegfs_sb.s_groups_count + STEGFS_DESC_PER_BLOCK(sb) - 1) /
		      STEGFS_DESC_PER_BLOCK(sb);
	for (i = 0; i < sb->u.stegfs_sb.s_groups_count; i++) {
		gdp = stegfs_get_group_desc (sb, i, NULL);
		if (!gdp)
			continue;
		desc_count += le16_to_cpu(gdp->bg_free_blocks_count);
		bitmap_nr = load_block_bitmap (sb, i);
		if (bitmap_nr < 0)
			continue;
		
		bh = sb->u.stegfs_sb.s_block_bitmap[bitmap_nr];

		if (!(sb->u.stegfs_sb.s_feature_ro_compat &
		     EXT2_FEATURE_RO_COMPAT_SPARSE_SUPER) ||
		    stegfs_group_sparse(i)) {
			if (!ext2_test_bit (0, bh->b_data))
				stegfs_error (sb, "stegfs_check_blocks_bitmap",
					    "Superblock in group %d "
					    "is marked free", i);

			for (j = 0; j < desc_blocks; j++)
				if (!ext2_test_bit (j + 1, bh->b_data))
					stegfs_error (sb,
					    "stegfs_check_blocks_bitmap",
					    "Descriptor block #%d in group "
					    "%d is marked free", j, i);
		}

		if (!block_in_use (le32_to_cpu(gdp->bg_block_bitmap), sb, bh->b_data))
			stegfs_error (sb, "stegfs_check_blocks_bitmap",
				    "Block bitmap for group %d is marked free",
				    i);

		if (!block_in_use (le32_to_cpu(gdp->bg_inode_bitmap), sb, bh->b_data))
			stegfs_error (sb, "stegfs_check_blocks_bitmap",
				    "Inode bitmap for group %d is marked free",
				    i);

		for (j = 0; j < sb->u.stegfs_sb.s_itb_per_group; j++)
			if (!block_in_use (le32_to_cpu(gdp->bg_inode_table) + j, sb, bh->b_data))
				stegfs_error (sb, "stegfs_check_blocks_bitmap",
					    "Block #%d of the inode table in "
					    "group %d is marked free", j, i);

		x = stegfs_count_free (bh, sb->s_blocksize);
		if (le16_to_cpu(gdp->bg_free_blocks_count) != x)
			stegfs_error (sb, "stegfs_check_blocks_bitmap",
				    "Wrong free blocks count for group %d, "
				    "stored = %d, counted = %lu", i,
				    le16_to_cpu(gdp->bg_free_blocks_count), x);
		bitmap_count += x;
	}
	if (le32_to_cpu(es->s_free_blocks_count) != bitmap_count)
		stegfs_error (sb, "stegfs_check_blocks_bitmap",
			    "Wrong free blocks count in super block, "
			    "stored = %lu, counted = %lu",
			    (unsigned long) le32_to_cpu(es->s_free_blocks_count), bitmap_count);
	unlock_super (sb);
}

int stegfs_blkused_ext2(struct super_block *sb, int bnum) {

	unsigned long gnum;
        struct buffer_head * bh;
	int bitmap_nr;

/*
	stegfs_debug("testing block: %d\n", bnum);
*/
	/* is it in use by ext2fs */
	gnum = (bnum-le32_to_cpu(sb->u.stegfs_sb.s_es->
				 s_first_data_block))
		/ STEGFS_SB(sb)->s_blocks_per_group;
	bitmap_nr = load_block_bitmap (sb, gnum);
	if (bitmap_nr < 0) {
		stegfs_debug("Error loading block bitmap\n");
		return 1;
	}
	
	bh = sb->u.stegfs_sb.s_block_bitmap[bitmap_nr];

	if (ext2_test_bit ((bnum - le32_to_cpu(sb->u.stegfs_sb.s_es->
					       s_first_data_block)) %
			   STEGFS_BLOCKS_PER_GROUP(sb), bh->b_data)) {
/*
		stegfs_debug("trying block %u - USED\n", bnum);
*/
		return 1;

	}

	return 0;
}

int stegfs_blkused(struct super_block *sb, int bnum) {

	unsigned long gnum;
        struct buffer_head * bh;
	int bitmap_nr, i;
	struct stegfs_btable btab, btab2;
	/*printk("*** stegfs_blkused called\n");*/
/*
	stegfs_debug("testing block: %d\n", bnum);
*/
	/* is it in use by ext2fs */
	gnum = (bnum-le32_to_cpu(sb->u.stegfs_sb.s_es->
				 s_first_data_block))
		/ STEGFS_SB(sb)->s_blocks_per_group;
	bitmap_nr = load_block_bitmap (sb, gnum);
	if (bitmap_nr < 0) {
		stegfs_debug("Error loading block bitmap\n");
		return 1;
	}
	
	bh = sb->u.stegfs_sb.s_block_bitmap[bitmap_nr];


	if (ext2_test_bit ((bnum - le32_to_cpu(sb->u.stegfs_sb.s_es->
					       s_first_data_block)) %
			   STEGFS_BLOCKS_PER_GROUP(sb), bh->b_data)) {
/*
		stegfs_debug("trying block %u - USED\n", bnum);
*/
		return 1;

	}

	stegfs_get_btab(sb, bnum, &btab);
	
	for(i=0; i<(STEGFS_MAX_LEVELS-1); i++) {
		if (STEGFS_SB(sb)->s_x->s_slkeys[i]) {
			/* try decrypting it with each key */
			stegfs_decrypt_btab2(sb, i+1, bnum, &btab, &btab2);
			if (btab2.magic1 == 0 && (btab2.magic2 == 0
						  || btab2.magic2 == 1) ) {
/*
				stegfs_debug("trying block %u - USED by sl %d (ino %lu)\n",
					     bnum, i+1, btab2.ino);
*/
				return 1;
			}
		}
	}
	/*
	stegfs_debug("trying block %u - FREE\n", bnum);
	*/
	return 0;
}

int stegfs_rand_new_block(struct super_block *sb) {

	unsigned int scale, maxblock;
	unsigned int randbs, randbsx;
	int k;

	maxblock = STEGFS_SB(sb)->s_es->s_blocks_count - 1;
	scale = ((unsigned int)-1) / maxblock;

	get_random_bytes(&randbs, sizeof(unsigned int));

	k=0;
	randbsx = randbs / scale;
	while (randbsx > maxblock ||
	       randbsx < 1 || stegfs_blkused(sb,randbsx)) {
		k++;
		if(k>10000) {
			return 0;
		}
		get_random_bytes(&randbs, sizeof(unsigned int));
		randbsx = randbs / scale;
	}
	return randbsx;
}
