#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <openssl/rand.h>
#include <openssl/sha.h>

#include "config.h"
#include "prngd.h"

static unsigned char obfuscator = 0;

 /*
  * rand_add(): wrapper for the OpenSSL RAND_add() function. The wrapper
  * is used to keep the calling functions simpler. Before adding entropy
  * to the pool, we want to prevent "chosen seed" attacks. Therefore, the
  * bytes obtained are not directly added, but they are "obfuscated" with
  * a more-or-less random value and then run through the SHA1-algorithm.
  * Hence, if an attacker would want to choose the seed-bytes, he would have
  * to construct the necessary SHA1-input(!) (which should be computationally
  * hard) _and_ predict the obfuscator-value.
  */
void rand_add(void *buf, int num, double entropy)
{
  int i, numbytes;
  unsigned char *uchar_buf;
  unsigned char sha_digest[SHA_DIGEST_LENGTH];

  uchar_buf = (unsigned char *)buf;

  for (i = 0; i < num; i++)
    uchar_buf[i] ^= obfuscator++;

  for (i = 0; i < num; i += SHA_DIGEST_LENGTH) {
    numbytes = num - i;
    if (numbytes > SHA_DIGEST_LENGTH)
      numbytes = SHA_DIGEST_LENGTH;
    SHA1(uchar_buf + i, numbytes, sha_digest);
    RAND_add(sha_digest, SHA_DIGEST_LENGTH, entropy * numbytes / num);
  }
}

 /*
  * rand_bytes(): wrapper around RAND_bytes(). Actually, all it does by now is
  * to stir around the "obfuscator" bits.
  */
int rand_bytes(unsigned char *buf, int num)
{
  int i, numbytes, retval;
  unsigned char randbyte;
  unsigned char mybuf[(PRNGD_STATE_SIZE * 2)];

  if (!RAND_status())
    return 0;		/* Not enough entropy in pool, yet */

  /*
   * Ok, we want to retrieve entropy. We want to make guessing the pool state
   * by continously retrieving random bytes more difficult, we perform to
   * additional steps, before actually retrieving the bytes:
   * - we mix the pool completely by querying PRNGD_STATE_SIZE bytes.
   * - we then position the read-pointer to another "random" position inside
   *   the pool.
   */
  if (!RAND_bytes(&randbyte, 1))
    return 0;
  numbytes = (int)(PRNGD_STATE_SIZE * (1.0 + (double) randbyte / 255));
  if (!RAND_bytes(mybuf, numbytes))
    return 0;
  for (i = 0; i < num; i++)
    obfuscator ^= mybuf[i];

  retval = RAND_bytes(buf, num);
  for (i = 0; i < num; i++)
    obfuscator ^= buf[i];

  return retval;
}

 /*
  * rand_pool(): wrapper around RAND_status(). Will either return 0 or 8192
  * for the amount of bits available in the pool.
  */
int rand_pool(void)
{
  if (RAND_status())
    return 8192;
  else
    return 0;
}
