<?php

declare(strict_types=1);

/*
 * Copyright (c) 2017-2022 François Kooman <fkooman@tuxed.net>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

namespace fkooman\SeCookie;

use fkooman\SeCookie\Exception\CookieException;

class CookieOptions
{
    private bool $secure = true;

    private ?string $path = null;

    private ?int $maxAge = null;

    private string $sameSite = 'Lax';

    public static function init(): self
    {
        return new self();
    }

    /**
     * Disable the "Secure" option. Use the *ONLY* for development.
     */
    public function withoutSecure(): self
    {
        $objCopy = clone $this;
        $objCopy->secure = false;

        return $objCopy;
    }

    public function withPath(string $path): self
    {
        $objCopy = clone $this;
        $objCopy->path = $path;

        return $objCopy;
    }

    public function withMaxAge(int $maxAge): self
    {
        if (0 >= $maxAge) {
            throw new CookieException('"MaxAge" must be positive');
        }

        $objCopy = clone $this;
        $objCopy->maxAge = $maxAge;

        return $objCopy;
    }

    public function withSameSiteNone(): self
    {
        $objCopy = clone $this;
        $objCopy->sameSite = 'None';

        return $objCopy;
    }

    public function withSameSiteLax(): self
    {
        $objCopy = clone $this;
        $objCopy->sameSite = 'Lax';

        return $objCopy;
    }

    public function withSameSiteStrict(): self
    {
        $objCopy = clone $this;
        $objCopy->sameSite = 'Strict';

        return $objCopy;
    }

    public function getSameSite(): string
    {
        return $this->sameSite;
    }

    /**
     * @return array<string>
     */
    public function attributeValueList(bool $deleteCookie, bool $dropSameSiteNone): array
    {
        $attributeValueList = [
            'HttpOnly',  // all cookies are ALWAYS "HttpOnly"
        ];
        if ($this->secure) {
            $attributeValueList[] = 'Secure';
        }
        if (null !== $path = $this->path) {
            $attributeValueList[] = sprintf('Path=%s', $path);
        }
        if (!$dropSameSiteNone) {
            $attributeValueList[] = sprintf('SameSite=%s', $this->sameSite);
        }
        if (null !== $maxAge = $this->determineMaxAge($deleteCookie)) {
            $attributeValueList[] = sprintf('Max-Age=%d', $maxAge);
        }

        sort($attributeValueList, SORT_STRING);

        return $attributeValueList;
    }

    private function determineMaxAge(bool $deleteCookie): ?int
    {
        if ($deleteCookie) {
            return 0;
        }

        return $this->maxAge;
    }
}
