#include <port_before.h>

#include <syslog.h>
#include <windows.h>
#include <snmp.h>
#include <oids.h>
#include <net/if.h>
#include <sys/socket.h>

#include <port_after.h>

typedef BOOL (WINAPI *pSnmpExtensionInit)(DWORD               dwTimeZeroReference,
										  HANDLE              *hPollForTrapEvent,
										  AsnObjectIdentifier *supportedView);

typedef BOOL (WINAPI *pSnmpExtensionQuery)(BYTE                   requestType,
										   OUT RFC1157VarBindList *variableBindings,
										   AsnInteger            *errorStatus,
										   AsnInteger            *errorIndex);

static HINSTANCE MIBInst;
static AsnObjectIdentifier MIB_NULL = {0,0};
static pSnmpExtensionInit		pInit;
static pSnmpExtensionQuery		pQuery;

#define IF_NAME		0
#define IF_ADDR		1
#define IF_STATE	2
#define IF_TYPE		3
#define IF_DEST		4
#define IF_BCAST	5

#define IP_ADDR_LEN	4

/*
 * Forward declarations
 */
BOOL LoadMIBExtensionDLL();
void UnloadMIBExtensionDLL();
BOOL Init();
BOOL Query(BYTE, RFC1157VarBindList *, AsnInteger *, AsnInteger *);

/*
 * Enumerate the TCP/IP interfaces on this machine.
 * Use SNMP to do this.
 *
 * It is the caller's responsibility to free elements
 * of IFData malloc'd by us.
 */
int GetInterfacesSNMP(struct ifdata *IFData[])
{
	int ifIndex, ifDataIndex, ret, len;
	char c;
	int numInterfaces = 0;
	struct in_addr addr, *dest;
	struct ifdata *ifd;

	AsnObjectIdentifier ifName		= { OIDSize(ifDescr), ifDescr};
	AsnObjectIdentifier ifKind		= { OIDSize(ifType), ifType};
	AsnObjectIdentifier ifStatus	= { OIDSize(ifOperStatus), ifOperStatus};
	AsnObjectIdentifier ifDest		= { OIDSize(tcpConnRemAddress), tcpConnRemAddress};
	AsnObjectIdentifier ifBcast		= { OIDSize(ipAdEntBcastAddr), ipAdEntBcastAddr};
	AsnObjectIdentifier ifAddr		= { OIDSize(ipAdEntAddr), ipAdEntAddr};

	AsnObjectIdentifier ifCount	= { OIDSize(ifNumber), ifNumber};
	
	RFC1157VarBindList  ifInfoBindList;
	RFC1157VarBind      ifInfoBind[6];

	RFC1157VarBindList  ifCountBindList;
	RFC1157VarBind      ifCountBind[1];
	
	AsnInteger          errorStatus;
	AsnInteger          errorIndex;
	
	ntPrintf(8, "Enumerating interfaces via SNMP\n");

	/* Load SNMP extension DLL */
	if(!LoadMIBExtensionDLL())
		return(0);
	
	/* Initialize SNMP */
	Init();

	/* Set up to query interfaces */
	ifInfoBindList.list = ifInfoBind;
	ifInfoBindList.len  = 6;
	ifInfoBind[IF_NAME].name	= MIB_NULL;
	ifInfoBind[IF_ADDR].name	= MIB_NULL;
	ifInfoBind[IF_STATE].name	= MIB_NULL;
	ifInfoBind[IF_TYPE].name	= MIB_NULL;
	ifInfoBind[IF_DEST].name	= MIB_NULL;
	ifInfoBind[IF_BCAST].name	= MIB_NULL;
	SNMP_oidcpy(&ifInfoBind[IF_NAME].name,	&ifName);
	SNMP_oidcpy(&ifInfoBind[IF_ADDR].name,	&ifAddr);
	SNMP_oidcpy(&ifInfoBind[IF_STATE].name,	&ifStatus);
	SNMP_oidcpy(&ifInfoBind[IF_TYPE].name,	&ifKind);
	SNMP_oidcpy(&ifInfoBind[IF_DEST].name,	&ifDest);
	SNMP_oidcpy(&ifInfoBind[IF_BCAST].name,	&ifBcast);
	
	/* Get number of interfaces present */
	ifCountBindList.list = ifCountBind;
	ifCountBindList.len  = 1;
	ifCountBind[0].name  = MIB_NULL;
	SNMP_oidcpy(&ifCountBind[0].name, &ifCount);

	ret = Query(ASN_RFC1157_GETNEXTREQUEST,&ifCountBindList,&errorStatus,&errorIndex);
	numInterfaces = ifCountBind[0].value.asnValue.number;
	SNMP_FreeVarBind(&ifCountBind[0]);
	
	if(numInterfaces > 0)
	{
		/* Now, stuff everything we have into ifdata structs */
		for(ifIndex = 0, ifDataIndex = 0; ifIndex < numInterfaces; ifIndex++)
		{
			Query(ASN_RFC1157_GETNEXTREQUEST, &ifInfoBindList, &errorStatus, &errorIndex);

			/* Address length is 4 for an IP address.  If not 4, not an IP addy (Sanity Check) */  
			if(ifInfoBind[IF_ADDR].value.asnValue.address.length != IP_ADDR_LEN)
				continue;

			/* Allocate the ifdata for this interface */
			ifd = (struct ifdata *)malloc(sizeof(struct ifdata));
			memset(ifd, 0, sizeof(struct ifdata));

			/* IP Address */
			addr.S_un.S_addr = *((long *)ifInfoBind[IF_ADDR].value.asnValue.address.stream);
			dest = &((struct sockaddr_in *)&ifd->if_addr)->sin_addr;
			memcpy(dest, &addr, sizeof(struct in_addr)); 
			ifd->if_addr.sa_family = AF_INET;

			/* Get the interface name */
			len = 0;
			/* Why do we do this?
			 * Because there are often nonprintable chars in this string for some reason.
			 * So we scan the string to get only the printable portion.
			 */
			while(len < ifInfoBind[IF_NAME].value.asnValue.string.length)
			{
				c = ifInfoBind[IF_NAME].value.asnValue.string.stream[len];
				if(c < 0x20 || c > 0x7E)
					break;
				len++;
			}
			memcpy(ifd->if_name, ifInfoBind[IF_NAME].value.asnValue.string.stream, len);

			/* Is interface up? */
			if(ifInfoBind[IF_STATE].value.asnValue.number == IF_UP)
				ifd->if_flags |= IFF_UP;
			
			/* Is it loopback? */
			if(ifInfoBind[IF_TYPE].value.asnValue.number == IF_LOOPBACK)
				ifd->if_flags |= IFF_LOOPBACK;

			/* Is it PPP? */
			if(ifInfoBind[IF_TYPE].value.asnValue.number == IF_PPP)
			{
				ifd->if_flags |= IFF_POINTTOPOINT;
			
				/* PPP Destination address */
				addr.S_un.S_addr = *((long *)ifInfoBind[IF_DEST].value.asnValue.address.stream);
				dest = &((struct sockaddr_in *)&ifd->if_dstaddr)->sin_addr;
				memcpy(dest, &addr, sizeof(struct in_addr)); 
				ifd->if_dstaddr.sa_family = AF_INET;
			}
			
			/* Do we have a broadcast address ?*/
			if(ifInfoBind[IF_BCAST].value.asnValue.address.length == IP_ADDR_LEN)
			{
				addr.S_un.S_addr = *((long *)ifInfoBind[IF_BCAST].value.asnValue.address.stream);
				dest = &((struct sockaddr_in *)&ifd->if_broadaddr)->sin_addr;
				memcpy(dest, &addr, sizeof(struct in_addr)); 
				ifd->if_addr.sa_family = AF_INET;
			}

			IFData[ifDataIndex] = ifd;
			ifDataIndex++;
		}
	}
	else
	{
		ntPrintf(1, "No interfaces found!\n");
	}
	/* Shutdown SNMP */
	SNMP_FreeVarBind(&ifInfoBind[0]);
	SNMP_FreeVarBind(&ifInfoBind[1]);
	SNMP_FreeVarBind(&ifInfoBind[2]);
	SNMP_FreeVarBind(&ifInfoBind[3]);
	SNMP_FreeVarBind(&ifInfoBind[4]);
	SNMP_FreeVarBind(&ifInfoBind[5]);
	UnloadMIBExtensionDLL();

	return((numInterfaces > 0) ? numInterfaces : 0);
}

/*
 * LoadMIBExtensionDLL()
 * Load the internet SNMP MIB DLL and get pointers to needed functions
 */
BOOL LoadMIBExtensionDLL()
{
	
	MIBInst = LoadLibrary("inetmib1.dll");
	if(MIBInst < (HINSTANCE) HINSTANCE_ERROR)
	{
		MIBInst = NULL;
		ntPrintf(1, "Could not load internet SNMP MIB DLL.  Cannot search for interfaces.\n");
		return(FALSE);
	}		
	pInit	=	(pSnmpExtensionInit)GetProcAddress(MIBInst ,"SnmpExtensionInit");
	pQuery	=	(pSnmpExtensionQuery)GetProcAddress(MIBInst ,"SnmpExtensionQuery");
	
	if(!pInit || !pQuery)
	{
		ntPrintf(1, "Could not find one or more required functions in the internet SNMP MIB DLL.  Cannot search for interfaces.\n");
		return(FALSE);
	}

	return(TRUE);
}

/*
 * LoadMIBExtensionDLL()
 * Release our reference to the SNMP MIB DLL
 */
void UnloadMIBExtensionDLL()
{
	if (MIBInst)
		FreeLibrary(MIBInst);
	MIBInst = NULL;
}


/*
 * LoadMIBExtensionDLL()
 * Initialize SNMP extensions
 */
BOOL Init()
{
	HANDLE pollForTrapEvent;
	AsnObjectIdentifier supportedView;
	return pInit(GetTickCount(),&pollForTrapEvent,&supportedView);
}

/*
 * LoadMIBExtensionDLL()
 * Query SNMP for information from the MIB
 */
BOOL Query(BYTE requestType, RFC1157VarBindList *variableBindings, AsnInteger *errorStatus,AsnInteger *errorIndex)
{
	return pQuery(requestType,variableBindings,errorStatus,errorIndex);
}
