/* vim: set sw=8: -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * fn-information.c:  Information built-in functions
 *
 * Authors:
 *  Jukka-Pekka Iivonen (iivonen@iki.fi)
 *  Jody Goldberg (jody@gnome.org)
 *  Morten Welinder (terra@diku.dk)
 *  Almer S. Tigelaar (almer@gnome.org)
 */
#include <gnumeric-config.h>
#include <gnumeric.h>
#include <func.h>

#include <parse-util.h>
#include <cell.h>
#include <str.h>
#include <sheet.h>
#include <workbook.h>
#include <format.h>
#include <formats.h>
#include <style.h>
#include <sheet-style.h>
#include <number-match.h>

#include <sys/utsname.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>

enum Value_Class {
	VALUE_CLASS_NUMBER  = 1,
	VALUE_CLASS_TEXT    = 2,
	VALUE_CLASS_BOOL    = 4,
	VALUE_CLASS_FORMULA = 8,
	VALUE_CLASS_ERROR   = 16,
	VALUE_CLASS_ARRAY   = 64,
	VALUE_CLASS_BOGUS   = -1
};


static enum Value_Class
get_value_class (FunctionEvalInfo *ei, ExprTree *expr)
{
	Value *value;
	enum Value_Class res;

	value = expr_eval (expr, ei->pos,
			   EVAL_PERMIT_NON_SCALAR|EVAL_PERMIT_EMPTY);
	if (value) {
		switch (value->type) {
		case VALUE_INTEGER:
		case VALUE_FLOAT:
			res = VALUE_CLASS_NUMBER;
			break;
		case VALUE_STRING:
			res = VALUE_CLASS_TEXT;
			break;
		case VALUE_BOOLEAN:
			res = VALUE_CLASS_BOOL;
			break;
		case VALUE_ERROR:
			res = VALUE_CLASS_ERROR;
			break;
		case VALUE_ARRAY:
			res = VALUE_CLASS_ARRAY;
			break;
		case VALUE_EMPTY:
		default:
			res = VALUE_CLASS_BOGUS;
			break;
		}
		value_release (value);
	} else
		res = VALUE_CLASS_ERROR;

	return res;
}

/***************************************************************************/

static char *help_cell = {
	N_("@FUNCTION=CELL\n"
	   "@SYNTAX=CELL(type,ref)\n"

	   "@DESCRIPTION="
	   "CELL returns information about the formatting, location, or "
	   "contents of a cell. "
	   "\n"
	   "@type specifies the type of information you want to obtain:\n "
	   "     address       Returns the given cell reference as text.\n"
	   "     col           Returns the number of the column in @ref.\n"
	   "     contents      Returns the contents of the cell in @ref.\n"
	   "     format        Returns the code of the format of the cell.\n"
	   "     parentheses   Returns 1 if @ref contains a negative value\n"
	   "                   and it's format displays it with parentheses.\n"
	   "     row           Returns the number of the row in @ref.\n"
	   "     width         Returns the column width.\n"
	   "\n"
           "This function is Excel compatible. "
	   "\n"
	   "@EXAMPLES=\n"
	   "CEll(\"format\",A1) returns the code of the format of the cell A1.\n"
	   "\n"
	   "@SEEALSO=")
};

typedef struct {
	char *format;
	char *output;
} translate_t;
static translate_t translate_table[] = {
	{ "General", "G" },
	{ "0", "F0" },
	{ "#,##0", ",0" },
	{ "0.00", "F2" },
	{ "#,##0.00", ",2" },
	{ "\"$\"#,##0_);\\(\"$\"#,##0\\)", "C0" },
	{ "\"$\"#,##0_);[Red]\\(\"$\"#,##0\\)", "C0-" },
	{ "\"$\"#,##0.00_);\\(\"$\"#,##0.00\\)", "C2" },
	{ "\"$\"#,##0.00_);[Red]\\(\"$\"#,##0.00\\)", "C2-" },
	{ "0%", "P0" },
	{ "0.00%", "P2" },
	{ "0.00e+00", "S2" },
	{ "# ?/?", "G" },
	{ "# ?" "?/?" "?", "G" },   /* Don't accidentally use trigraphs here.  */
	{ "m/d/yy", "D4" },
	{ "m/d/yy h:mm", "D4" },
	{ "mm/dd/yy", "D4" },
	{ "d-mmm-yy", "D1" },
	{ "dd-mmm-yy", "D1" },
	{ "d-mmm", "D2" },
	{ "dd-mmm", "D2" },
	{ "mmm-yy", "D3" },
	{ "mm/dd", "D5" },
	{ "h:mm am/pm", "D7" },
	{ "h:mm:ss am/pm", "D6" },
	{ "h:mm", "D9" },
	{ "h:mm:ss", "D8" }
};

static Value *
translate_cell_format (StyleFormat const *format)
{
	int i;
	char *fmt;
	const int translate_table_count = sizeof (translate_table) / sizeof(translate_t);

	if (format == NULL)
		return value_new_string ("G");

	fmt = style_format_as_XL (format, FALSE);

	/*
	 * TODO : What does this do in different locales ??
	 */
	for (i = 0; i < translate_table_count; i++) {
		const translate_t *t = &translate_table[i];

		if (!g_strcasecmp (fmt, t->format)) {
			g_free (fmt);
			return value_new_string (t->output);
		}
	}

	g_free (fmt);
	return value_new_string ("G");
}

static FormatCharacteristics
retrieve_format_info (Sheet *sheet, int col, int row)
{
	MStyle *mstyle = sheet_style_get (sheet, col, row);
	StyleFormat *format = mstyle_get_format (mstyle);
	FormatCharacteristics info;

	cell_format_classify (format, &info);

	return info;
}

static Value *
gnumeric_cell (FunctionEvalInfo *ei, Value **argv)
{
	const char *info_type = value_peek_string (argv[0]);
	CellRef ref = argv [1]->v_range.cell.a;

	if (!g_strcasecmp(info_type, "address")) {
		/* Reference of the first cell in reference, as text. */
		return value_new_string (cell_coord_name (ref.col, ref.row));
	} else if (!g_strcasecmp (info_type, "col")) {
		return value_new_int (ref.col + 1);
	} else if (!g_strcasecmp (info_type, "color")) {
		FormatCharacteristics info = retrieve_format_info (ei->pos->sheet,
								   ref.col,
								   ref.row);

		/* 0x01 = first bit (1) indicating negative colors */
		return (info.negative_fmt & 0x01) ? value_new_int (1) :
			value_new_int (0);
	} else if (!g_strcasecmp (info_type, "contents")) {
		Cell *cell = sheet_cell_get (ei->pos->sheet, ref.col, ref.row);

		if (cell && cell->value)
			return value_duplicate (cell->value);
		return value_new_empty ();
	} else if (!g_strcasecmp (info_type, "filename"))	{
		char *name = ei->pos->sheet->workbook->filename;

		if (name == NULL)
			return value_new_string ("");
		else
			return value_new_string (name);
	} else if (!g_strcasecmp (info_type, "format")) {
		MStyle *mstyle = sheet_style_get (ei->pos->sheet, ref.col,
						  ref.row);

		return translate_cell_format (mstyle_get_format (mstyle));
	} else if (!g_strcasecmp (info_type, "parentheses")) {
		FormatCharacteristics info = retrieve_format_info (ei->pos->sheet,
								   ref.col,
								   ref.row);

		/* 0x02 = second bit (2) indicating parentheses */
		return (info.negative_fmt & 0x02) ? value_new_int (1) :
			value_new_int (0);
	} else if (!g_strcasecmp (info_type, "prefix")) {
		MStyle *mstyle = sheet_style_get (ei->pos->sheet, ref.col,
						  ref.row);
		Cell *cell = sheet_cell_get (ei->pos->sheet, ref.col, ref.row);

		if (cell && cell->value && cell->value->type == VALUE_STRING) {
			switch (mstyle_get_align_h (mstyle)) {
			case HALIGN_GENERAL: return value_new_string ("'");
			case HALIGN_LEFT:    return value_new_string ("'");
			case HALIGN_RIGHT:   return value_new_string ("\"");
			case HALIGN_CENTER:  return value_new_string ("^");
			case HALIGN_FILL:    return value_new_string ("\\");
			default : 	     return value_new_string ("");
			}
		}
		return value_new_string ("");
	} else if (!g_strcasecmp (info_type, "protect")) {
		MStyle const *mstyle = sheet_style_get (ei->pos->sheet, ref.col,
							ref.row);
		return value_new_int (mstyle_get_content_locked (mstyle) ? 1 : 0);
	} else if (!g_strcasecmp (info_type, "row")) {
		return value_new_int (ref.row + 1);
	} else if (!g_strcasecmp (info_type, "type")) {
		Cell *cell;

		cell = sheet_cell_get (ei->pos->sheet, ref.col, ref.row);
		if (cell && cell->value) {
			if (cell->value->type == VALUE_STRING)
				return value_new_string ("l");
			else
				return value_new_string ("v");
		}
		return value_new_string ("b");
	} else if (!g_strcasecmp (info_type, "width")) {
		ColRowInfo const *info = sheet_col_get_info (ei->pos->sheet, ref.col);
		double charwidth;
		int    cellwidth;

		charwidth = gnumeric_default_font->approx_width.pts;
		cellwidth = info->size_pts;

		return value_new_int (rint (cellwidth / charwidth));
	}

	return value_new_error (ei->pos, _("Unknown info_type"));
}

/***************************************************************************/

static char *help_expression = {
	N_("@FUNCTION=EXPRESSION\n"
	   "@SYNTAX=EXPRESSION(cell)\n"
	   "@DESCRIPTION="
	   "EXPRESSION returns expression in @cell as a string, or "
	   "empty if the cell is not an expression.\n"
	   "@EXAMPLES=\n"
	   "in A1 EXPRESSION(A2) equals 'EXPRESSION(A3)'.\n"
	   "in A2 EXPRESSION(A3) equals empty.\n"
	   "\n"
	   "@SEEALSO=TEXT")
};

static Value *
gnumeric_expression (FunctionEvalInfo *ei, Value **args)
{
	Value const * const v = args[0];
	if (v->type == VALUE_CELLRANGE) {
		Cell *cell;
		CellRef const * a = &v->v_range.cell.a;
		CellRef const * b = &v->v_range.cell.b;

		if (a->col != b->col || a->row != b->row || a->sheet !=b->sheet)
			return value_new_error (ei->pos, gnumeric_err_REF);

		cell = sheet_cell_get (eval_sheet (a->sheet, ei->pos->sheet),
				       a->col, a->row);

		if (cell && cell_has_expr (cell)) {
			ParsePos pos;
			char * expr_string =
			    expr_tree_as_string (cell->base.expression,
				parse_pos_init_cell (&pos, cell));
			Value * res = value_new_string (expr_string);
			g_free (expr_string);
			return res;
		}
	}

	return value_new_empty ();
}


/***************************************************************************/

static char *help_countblank = {
        N_("@FUNCTION=COUNTBLANK\n"
           "@SYNTAX=COUNTBLANK(range)\n"

           "@DESCRIPTION="
           "COUNTBLANK returns the number of blank cells in a @range.\n"
	   "This function is Excel compatible. "
           "\n"
	   "@EXAMPLES=\n"
	   "COUNTBLANK(A1:A20) returns the number of blank cell in A1:A20.\n"
	   "\n"
           "@SEEALSO=COUNT")
};

static Value *
cb_countblank (Sheet *sheet, int col, int row,
	       Cell *cell, void *user_data)
{
	if (!cell_is_blank (cell))
		*((int *)user_data) -= 1;
	return NULL;
}

static Value *
gnumeric_countblank (FunctionEvalInfo *ei, Value **args)
{
	/* FIXME : This does not handle 3D references */
	int count =
		value_area_get_height (ei->pos, args[0]) *
		value_area_get_width (ei->pos, args[0]);

	workbook_foreach_cell_in_range (ei->pos, args[0], TRUE,
					&cb_countblank, &count);

	return value_new_int (count);
}

/***************************************************************************/

static char *help_info = {
	N_("@FUNCTION=INFO\n"
	   "@SYNTAX=INFO(type)\n"

	   "@DESCRIPTION="
	   "INFO returns information about the current operating environment. "
	   "\n"
	   "@type is the type of information you want to obtain:\n"
	   "    memavail        Returns the amount of memory available (bytes).\n"
	   "    memused         Returns the amount of memory used (bytes).\n"
	   "    numfile         Returns the number of active worksheets.\n"
	   "    osversion       Returns the operating system version.\n"
	   "    recalc          Returns the recalculation mode (automatic).\n"
	   "    release         Returns the version of Gnumeric as text.\n"
	   "    system          Returns the name of the environment.\n"
	   "    totmem          Returns the amount of total memory available.\n"
	   "\n"
	   "This function is Excel compatible, except that types directory "
	   "and origin are not implemented. "
	   "\n"
	   "@EXAMPLES=\n"
	   "INFO(\"system\") returns \"Linux\" on a Linux system.\n"
	   "\n"
	   "@SEEALSO=")
};


static Value *
gnumeric_info (FunctionEvalInfo *ei, Value **argv)
{
	char const * const info_type = value_peek_string (argv[0]);
	if (!g_strcasecmp (info_type, "directory")) {
		/* Path of the current directory or folder.  */
		return value_new_error (ei->pos, _("Unimplemented"));
	} else if (!g_strcasecmp (info_type, "memavail")) {
		/* Amount of memory available, in bytes.  */
		return value_new_int (15 << 20);  /* Good enough... */
	} else if (!g_strcasecmp (info_type, "memused")) {
		/* Amount of memory being used for data.  */
		return value_new_int (1 << 20);  /* Good enough... */
	} else if (!g_strcasecmp (info_type, "numfile")) {
		/* Number of active worksheets.  */
		return value_new_int (1);  /* Good enough... */
	} else if (!g_strcasecmp (info_type, "origin")) {
		/* Absolute A1-style reference, as text, prepended with "$A:"
		 * for Lotus 1-2-3 release 3.x compatibility. Returns the cell
		 * reference of the top and leftmost cell visible in the
		 * window, based on the current scrolling position.
		 */
		return value_new_error (ei->pos, _("Unimplemented"));
	} else if (!g_strcasecmp (info_type, "osversion")) {
		/* Current operating system version, as text.  */
		struct utsname unamedata;

		if (uname (&unamedata) == -1)
			return value_new_error (ei->pos,
						_("Unknown version"));
		else {
			char *tmp = g_strdup_printf (_("%s version %s"),
						     unamedata.sysname,
						     unamedata.release);
			Value *res = value_new_string (tmp);
			g_free (tmp);
			return res;
		}
	} else if (!g_strcasecmp (info_type, "recalc")) {
		/* Current recalculation mode; returns "Automatic" or "Manual".  */
		return value_new_string (_("Automatic"));
	} else if (!g_strcasecmp (info_type, "release")) {
		/* Version of Gnumeric (Well, Microsoft Excel), as text.  */
		return value_new_string (GNUMERIC_VERSION);
	} else if (!g_strcasecmp (info_type, "system")) {
		/* Name of the operating environment.  */
		struct utsname unamedata;

		if (uname (&unamedata) == -1)
			return value_new_error (ei->pos, _("Unknown system"));
		else
			return value_new_string (unamedata.sysname);
	} else if (!g_strcasecmp (info_type, "totmem")) {
		/* Total memory available, including memory already in use, in
		 * bytes.
		 */
		return value_new_int (16 << 20);  /* Good enough... */
	}

	return value_new_error (ei->pos, _("Unknown info_type"));
}

/***************************************************************************/

static char *help_iserror = {
	N_("@FUNCTION=ISERROR\n"
	   "@SYNTAX=ISERROR(value)\n"

	   "@DESCRIPTION="
	   "ISERROR returns a TRUE value if the expression has an error.\n"
	   "This function is Excel compatible. "
	   "\n"
	   "@EXAMPLES=\n"
	   "ISERROR(NA()) equals TRUE.\n"
	   "\n"
	   "@SEEALSO=ERROR")
};

/* A utility routine to evaluate a single argument and return any errors
 * directly
 */
static Value *
gnumeric_check_for_err (FunctionEvalInfo *ei, ExprList *expr_node_list,
			Value ** err)
{
	Value * tmp;

	if (expr_list_length (expr_node_list) != 1) {
		*err = value_new_error(ei->pos,
				       _("Argument mismatch"));
		return NULL;
	}
	tmp = expr_eval (expr_node_list->data, ei->pos, EVAL_STRICT);

	if (tmp != NULL) {
		if (tmp->type == VALUE_ERROR)
			return tmp;
		value_release (tmp);
	}
	return NULL;
}

static Value *
gnumeric_iserror (FunctionEvalInfo *ei, ExprList *expr_node_list)
{
	Value * res, *err = NULL;
	res = gnumeric_check_for_err (ei, expr_node_list, &err);
	if (err != NULL)
		return err;

	if (res) {
		value_release (res);
		return value_new_bool (TRUE);
	} else
		return value_new_bool (FALSE);
}

/***************************************************************************/

static char *help_isna = {
	N_("@FUNCTION=ISNA\n"
	   "@SYNTAX=ISNA(value)\n"

	   "@DESCRIPTION="
	   "ISNA returns TRUE if the value is the #N/A error value.\n"
	   "This function is Excel compatible. "
	   "\n"
	   "@EXAMPLES=\n"
	   "ISNA(NA()) equals TRUE.\n"
	   "\n"
	   "@SEEALSO=NA")
};

/*
 * We need to operator directly in the input expression in order to bypass
 * the error handling mechanism
 */
static Value *
gnumeric_isna (FunctionEvalInfo *ei, ExprList *expr_node_list)
{
	Value * res, *err = NULL;
	gboolean b;

	res = gnumeric_check_for_err (ei, expr_node_list, &err);
	if (err != NULL)
		return err;

	b = (res && !strcmp (gnumeric_err_NA, res->v_err.mesg->str));
	if (res) value_release (res);
	return value_new_bool (b);
}

/***************************************************************************/

static char *help_iserr = {
	N_("@FUNCTION=ISERR\n"
	   "@SYNTAX=ISERR(value)\n"

	   "@DESCRIPTION="
	   "ISERR returns TRUE if the value is any error value except #N/A.\n"
	   "This function is Excel compatible. "
	   "\n"
	   "@EXAMPLES=\n"
	   "ISERR(NA()) return FALSE.\n"
	   "\n"
	   "@SEEALSO=ISERROR")
};

static Value *
gnumeric_iserr (FunctionEvalInfo *ei, ExprList *expr_node_list)
{
	Value * res, *err = NULL;
	gboolean b;

	res = gnumeric_check_for_err (ei, expr_node_list, &err);
	if (err != NULL)
		return err;

	b = (res && strcmp (gnumeric_err_NA, res->v_err.mesg->str));
	if (res) value_release (res);
	return value_new_bool (b);
}

/***************************************************************************/

static char *help_error_type = {
	N_("@FUNCTION=ERROR.TYPE\n"
	   "@SYNTAX=ERROR(value)\n"

	   "@DESCRIPTION="
	   "ERROR.TYPE returns an error number corresponding to the given "
	   "error value.  The error numbers for error values are\n"
	   "#DIV/0!    2\n"
	   "#VALUE!    3\n"
	   "#REF!      4\n"
	   "#NAME?     5\n"
	   "#NUM!      6\n"
	   "#N/A       7\n"
	   "This function is Excel compatible. "
	   "\n"
	   "@EXAMPLES=\n"
	   "ERROR.TYPE(NA()) equals 7.\n"
	   "\n"
	   "@SEEALSO=ISERROR")
};

static Value *
gnumeric_error_type (FunctionEvalInfo *ei, ExprList *expr_node_list)
{
	int retval = -1;
	char const * mesg;
	Value * res, *err = NULL;
	res = gnumeric_check_for_err (ei, expr_node_list, &err);
	if (err != NULL)
		return err;
	if (res == NULL)
		return value_new_error (ei->pos, gnumeric_err_NA);

	mesg = res->v_err.mesg->str;
	if (!strcmp (gnumeric_err_NULL, mesg))
		retval = 1;
	else if (!strcmp (gnumeric_err_DIV0, mesg))
		retval = 2;
	else if (!strcmp (gnumeric_err_VALUE, mesg))
		retval = 3;
	else if (!strcmp (gnumeric_err_REF, mesg))
		retval = 4;
	else if (!strcmp (gnumeric_err_NAME, mesg))
		retval = 5;
	else if (!strcmp (gnumeric_err_NUM, mesg))
		retval = 6;
	else if (!strcmp (gnumeric_err_NA, mesg))
		retval = 7;
	else {
		value_release (res);
		return value_new_error (ei->pos, gnumeric_err_NA);
	}

	value_release (res);
	return value_new_int (retval);
}

/***************************************************************************/

static char *help_na = {
	N_("@FUNCTION=NA\n"
	   "@SYNTAX=NA()\n"

	   "@DESCRIPTION="
	   "NA returns the error value #N/A.\n"
	   "This function is Excel compatible. "
	   "\n"
	   "@EXAMPLES=\n"
	   "NA() equals #N/A error.\n"
	   "\n"
	   "@SEEALSO=ISNA")
};

static Value *
gnumeric_na (FunctionEvalInfo *ei, Value **argv)
{
	return value_new_error (ei->pos, gnumeric_err_NA);
}

/***************************************************************************/

static char *help_error = {
	N_("@FUNCTION=ERROR\n"
	   "@SYNTAX=ERROR(text)\n"

	   "@DESCRIPTION="
	   "ERROR return the specified error.\n"
	   "\n"
	   "@EXAMPLES=\n"
	   "ERROR(\"#OWN ERROR\").\n"
	   "\n"
	   "@SEEALSO=ISERROR")
};

static Value *
gnumeric_error (FunctionEvalInfo *ei, Value *argv[])
{
	return value_new_error (ei->pos, value_peek_string (argv[0]));
}

/***************************************************************************/

static char *help_isblank = {
	N_("@FUNCTION=ISBLANK\n"
	   "@SYNTAX=ISBLANK(value)\n"

	   "@DESCRIPTION="
	   "ISBLANK returns TRUE if the value is blank.\n"
	   "This function is Excel compatible. "
	   "\n"
	   "@EXAMPLES=\n"
	   "ISBLANK(A1).\n"
	   "\n"
	   "@SEEALSO=")
};

static Value *
gnumeric_isblank (FunctionEvalInfo *ei, ExprList *expr_node_list)
{
	gboolean result = FALSE;
	ExprTree *expr;
	if (expr_list_length (expr_node_list) != 1)
		return value_new_error (ei->pos,
					_("Invalid number of arguments"));

	expr = expr_node_list->data;

	/* How can this happen ? */
	if (expr == NULL)
		return value_new_bool (FALSE);

	/* Handle pointless arrays */
	if (expr->any.oper == OPER_ARRAY) {
		if (expr->array.rows != 1 || expr->array.cols != 1)
			return value_new_bool (FALSE);
		expr = expr->array.corner.expr;
	}

	if (expr->any.oper == OPER_VAR) {
		CellRef const *ref = &expr->var.ref;
		Sheet const *sheet = eval_sheet (ref->sheet, ei->pos->sheet);
		CellPos pos;

		cellref_get_abs_pos (ref, &ei->pos->eval, &pos);
		result = cell_is_blank (sheet_cell_get (sheet, pos.col, pos.row));
	}
	return value_new_bool (result);
}

/***************************************************************************/

static char *help_iseven = {
	N_("@FUNCTION=ISEVEN\n"
	   "@SYNTAX=ISEVEN(value)\n"

	   "@DESCRIPTION="
	   "ISEVEN returns TRUE if the number is even.\n"
	   "This function is Excel compatible. "
	   "\n"
	   "@EXAMPLES=\n"
	   "ISEVEN(4) equals TRUE.\n"
	   "\n"
	   "@SEEALSO=ISODD")
};

static Value *
gnumeric_iseven (FunctionEvalInfo *ei, Value **argv)
{
	return value_new_bool (!(value_get_as_int (argv[0]) & 1));
}

/***************************************************************************/

static char *help_islogical = {
	N_("@FUNCTION=ISLOGICAL\n"
	   "@SYNTAX=ISLOGICAL(value)\n"

	   "@DESCRIPTION="
	   "ISLOGICAL returns TRUE if the value is a logical value.\n"
	   "This function is Excel compatible. "
	   "\n"
	   "@EXAMPLES=\n"
	   "ISLOGICAL(A1).\n"
	   "\n"
	   "@SEEALSO=")
};

static Value *
gnumeric_islogical (FunctionEvalInfo *ei, ExprList *expr_node_list)
{
	enum Value_Class cl;

	if (expr_list_length (expr_node_list) != 1)
		return value_new_error (ei->pos,
					_("Invalid number of arguments"));

	cl = get_value_class (ei, expr_node_list->data);

	return value_new_bool (cl == VALUE_CLASS_BOOL);
}

/***************************************************************************/

static char *help_isnontext = {
	N_("@FUNCTION=ISNONTEXT\n"
	   "@SYNTAX=ISNONTEXT(value)\n"

	   "@DESCRIPTION="
	   "ISNONTEXT Returns TRUE if the value is not text.\n"
	   "This function is Excel compatible. "
	   "\n"
	   "@EXAMPLES=\n"
	   "ISNONTEXT(\"text\") equals FALSE.\n"
	   "\n"
	   "@SEEALSO=ISTEXT")
};

static Value *
gnumeric_isnontext (FunctionEvalInfo *ei, ExprList *expr_node_list)
{
	if (expr_list_length (expr_node_list) != 1)
		return value_new_error (ei->pos,
					_("Invalid number of arguments"));

	return value_new_bool (get_value_class (ei, expr_node_list->data)
			       != VALUE_CLASS_TEXT);
}

/***************************************************************************/

static char *help_isnumber = {
	N_("@FUNCTION=ISNUMBER\n"
	   "@SYNTAX=ISNUMBER(value)\n"

	   "@DESCRIPTION="
	   "ISNUMBER returns TRUE if the value is a number.\n"
	   "This function is Excel compatible. "
	   "\n"
	   "@EXAMPLES=\n"
	   "ISNUMBER(\"text\") equals FALSE.\n"
	   "\n"
	   "@SEEALSO=")
};

static Value *
gnumeric_isnumber (FunctionEvalInfo *ei, ExprList *expr_node_list)
{
	if (expr_list_length (expr_node_list) != 1)
		return value_new_error (ei->pos,
					_("Invalid number of arguments"));

	return value_new_bool (get_value_class (ei, expr_node_list->data)
			       == VALUE_CLASS_NUMBER);
}

/***************************************************************************/

static char *help_isodd = {
	N_("@FUNCTION=ISODD\n"
	   "@SYNTAX=ISODD(value)\n"

	   "@DESCRIPTION="
	   "ISODD returns TRUE if the number is odd.\n"
	   "This function is Excel compatible. "
	   "\n"
	   "@EXAMPLES=\n"
	   "ISODD(3) equals TRUE.\n"
	   "\n"
	   "@SEEALSO=ISEVEN")
};

static Value *
gnumeric_isodd (FunctionEvalInfo *ei, Value **argv)
{
	return value_new_bool (value_get_as_int (argv[0]) & 1);
}

/***************************************************************************/

static char *help_isref = {
	N_("@FUNCTION=ISREF\n"
	   "@SYNTAX=ISREF(value)\n"

	   "@DESCRIPTION="
	   "ISREF returns TRUE if the value is a reference.\n"
	   "This function is Excel compatible. "
	   "\n"
	   "@EXAMPLES=\n"
	   "ISREF(A1) equals TRUE.\n"
	   "\n"
	   "@SEEALSO=")
};

static Value *
gnumeric_isref (FunctionEvalInfo *ei, ExprList *expr_node_list)
{
	ExprTree *t;

	if (expr_list_length (expr_node_list) != 1)
		return value_new_error (ei->pos,
					_("Invalid number of arguments"));

	t = expr_node_list->data;
	if (!t)
		return NULL;

	return value_new_bool (t->any.oper == OPER_VAR);
}

/***************************************************************************/

static char *help_istext = {
	N_("@FUNCTION=ISTEXT\n"
	   "@SYNTAX=ISTEXT(value)\n"

	   "@DESCRIPTION="
	   "ISTEXT returns TRUE if the value is text.\n"
	   "This function is Excel compatible. "
	   "\n"
	   "@EXAMPLES=\n"
	   "ISTEXT(\"text\") equals TRUE.\n"
	   "\n"
	   "@SEEALSO=ISNONTEXT")
};

static Value *
gnumeric_istext (FunctionEvalInfo *ei, ExprList *expr_node_list)
{
	if (expr_list_length (expr_node_list) != 1)
		return value_new_error (ei->pos,
					_("Invalid number of arguments"));

	return value_new_bool (get_value_class (ei, expr_node_list->data)
			       == VALUE_CLASS_TEXT);
}

/***************************************************************************/

static char *help_n = {
	N_("@FUNCTION=N\n"
	   "@SYNTAX=N(value)\n"

	   "@DESCRIPTION="
	   "N returns a value converted to a number.  Strings containing "
	   "text are converted to the zero value.\n"
	   "This function is Excel compatible. "
	   "\n"
	   "@EXAMPLES=\n"
	   "N(\"42\") equals 42.\n"
	   "\n"
	   "@SEEALSO=")
};

static Value *
gnumeric_n (FunctionEvalInfo *ei, Value **argv)
{
	const char *str;
	Value *v;

	if (argv[0]->type == VALUE_BOOLEAN)
		return value_new_int (value_get_as_int(argv[0]));

	if (VALUE_IS_NUMBER (argv[0]))
		return value_duplicate (argv[0]);

	if (argv[0]->type != VALUE_STRING)
		return value_new_error (ei->pos, gnumeric_err_NUM);

	str = value_peek_string (argv[0]);
	v = format_match_number (str, NULL);
	if (v != NULL)
		return v;
	return value_new_float (0);
}

/***************************************************************************/

static char *help_type = {
	N_("@FUNCTION=TYPE\n"
	   "@SYNTAX=TYPE(value)\n"

	   "@DESCRIPTION="
	   "TYPE returns a number indicating the data type of a value.\n"
	   "This function is Excel compatible. "
	   "\n"
	   "@EXAMPLES=\n"
	   "TYPE(3) equals 1.\n"
	   "TYPE(\"text\") equals 2.\n"
	   "\n"
	   "@SEEALSO=")
};

static Value *
gnumeric_type (FunctionEvalInfo *ei, ExprList *expr_node_list)
{
	if (expr_list_length (expr_node_list) != 1)
		return value_new_error (ei->pos,
					_("Invalid number of arguments"));

	return value_new_int (get_value_class (ei, expr_node_list->data));
}

/***************************************************************************/

static char *help_getenv = {
	N_("@FUNCTION=GETENV\n"
	   "@SYNTAX=GETENV(string)\n"

	   "@DESCRIPTION="
	   "GETENV retrieves a value from the execution environment.\n"
	   "\n"
	   "If the variable specified by @STRING does not exist, #N/A! will "
	   "be returned.  Note, that variable names are case sensitive.\n"
	   "@EXAMPLES=\n"
	   "\n"
	   "@SEEALSO=")
};

static Value *
gnumeric_getenv (FunctionEvalInfo *ei, Value **argv)
{
	const char *var = value_peek_string (argv[0]);
	const char *val = getenv (var);

	if (val)
		return value_new_string (val);
	else
		return value_new_error (ei->pos, gnumeric_err_NA);
}

/***************************************************************************/

void information_functions_init (void);

void
information_functions_init (void)
{
	FunctionCategory *cat = function_get_category_with_translation ("Information", _("Information"));

	function_add_args  (cat, "cell", "sr", "info_type, cell",
			    &help_cell, gnumeric_cell);
        function_add_args  (cat, "countblank", "r",  "range",
			    &help_countblank, gnumeric_countblank);
	function_add_args  (cat, "error",   "s",  "text",
			    &help_error,   gnumeric_error);
	function_add_nodes (cat, "error.type", NULL, "",
			    &help_error_type, gnumeric_error_type);
	function_add_args  (cat, "expression", "r",   "cell",
			    &help_expression, gnumeric_expression);
	function_add_args  (cat, "info", "s", "info_type",
			    &help_info, gnumeric_info);
	function_add_nodes (cat, "isblank", NULL, "value",
			    &help_isblank, gnumeric_isblank);
	function_add_nodes (cat, "iserr", NULL,   "",
			    &help_iserr,   gnumeric_iserr);
	function_add_nodes (cat, "iserror", NULL,   "",
			    &help_iserror, gnumeric_iserror);
	function_add_args  (cat, "iseven", "?", "value",
			    &help_iseven, gnumeric_iseven);
	function_add_nodes (cat, "islogical", NULL, "value",
			    &help_islogical, gnumeric_islogical);
	function_add_nodes (cat, "isna", NULL,   "",
			    &help_isna,    gnumeric_isna);
	function_add_nodes (cat, "isnontext", NULL, "value",
			    &help_isnontext, gnumeric_isnontext);
	function_add_nodes (cat, "isnumber", NULL, "value",
			    &help_isnumber, gnumeric_isnumber);
	function_add_args  (cat, "isodd", "?", "value",
			    &help_isodd, gnumeric_isodd);
	function_add_nodes (cat, "isref", NULL, "value",
			    &help_isref, gnumeric_isref);
	function_add_nodes (cat, "istext", NULL, "value",
			    &help_istext, gnumeric_istext);
	function_add_args  (cat, "n", "?", "value",
			    &help_n, gnumeric_n);
	function_add_args  (cat, "na",      "",  "",
			    &help_na,      gnumeric_na);
	function_add_nodes (cat, "type",   NULL, "value",
			    &help_type, gnumeric_type);
	function_add_args  (cat, "getenv", "s", "string",
			    &help_getenv, gnumeric_getenv);
}
